#include "speechdetectionandseparation.h"


// Create a hamming window of windowLength samples in buffer
void speechdetectionandseparation::hammingwindow(int windowLength) {
	for (int i = 0; i < windowLength; i++) {
		hammingwin[i] = 1.0; // (0.53836 - (0.46164 * cos(2 * PI * ((float)i / (((float)windowLength) * 1.0)))));
	}
}

speechdetectionandseparation::speechdetectionandseparation(char microphone_array_geometry[], int blocksize)
{

	// parsing and serializing JSON
	json j_complete = json::parse(microphone_array_geometry);

	nofmicrophones = j_complete["NumberofMicrophones"];
	array_type = (ArrayType_sds)j_complete["MicrophoneArrayType"];
	array_scattering_type = (ArrayScattering_sds)j_complete["MicrophoneArrayScat"];
	sampling_rate = samplingrates[static_cast<int>(j_complete["SamplingRate"])];
	sample_type = sampletypes[static_cast<int>(j_complete["SampleType"])];
	block_size = blocksizes[static_cast<int>(j_complete["BlockSize"])];

	// 10 is the max time index with 1024 frequency index
	AudioBlockSize = blocksize;
	MaxNumberofSeparatedObjects = (int)MaxNumberofObjects;
	NumberofFrequencybins = block_size / 2;
	
	ifstream micarray_parameters((string)j_complete["MicrophoneArrayFilterURI"]);
	string param_text;
	for (int gl = 0; gl < 4; gl++)
	{
		getline(micarray_parameters, param_text);
		hp_kernel->setmicarray_parameters(gl, param_text);
	}

	micarray_parameters.close();

	micarr = micarray_sds();

	for (int klm = 0; klm < nofmicrophones; klm++)
	{
		micarr.addnewmicrophone(j_complete["MicrophoneList"][klm]["xCoord"], j_complete["MicrophoneList"][klm]["yCoord"], j_complete["MicrophoneList"][klm]["zCoord"], j_complete["MicrophoneList"][klm]["micxLookCoord"], j_complete["MicrophoneList"][klm]["micyLookCoord"], j_complete["MicrophoneList"][klm]["miczLookCoord"], j_complete["MicrophoneArrayLookCoord"]["xLookCoord"], j_complete["MicrophoneArrayLookCoord"]["yLookCoord"], j_complete["MicrophoneArrayLookCoord"]["zLookCoord"]);
	}

	hp_kernel = new healpixkernel_sds(HEALPixOrder);

	// Get the number of sampling points
	healpix_samplingpoints = hp_kernel->gethealpixnumberofpixels();

	// It is the dimansion used for the calculation of the SRF vector
	int srf_transformation_size = nofmicrophones * healpix_samplingpoints * MaxSphericalHarmonicsOrder;
	// Histogram size 
	int hist_size_forblocks = AudioBlockSize * NumberofFrequencybins;


	// Get the microphone array angles
	micanglestheta = micarr.getmicthetas();
	micanglesphi = micarr.getmicphis();

	// Calculate the legendre kernels with the specified spherical harmonics order
// The format is in N*[Pixelsize*Pixelsize] where N is the maximum spherical harmonics order
// The imaginary parts of the dictionary kernels
	magkernels = hp_kernel->calculatelegendrekernels(MaxSphericalHarmonicsOrder);
	hp_kernel->calculateorthogonalLK();

	//multiplicationres = (float*)(hp_kernel->getmultiplicationresult());

	// Transformation matrices from the microphone array measurements to Steered Response Function
	kr_1_real = (float*)calloc((int)(srf_transformation_size), sizeof(float));
	kr_1_imag = (float*)calloc((int)(srf_transformation_size), sizeof(float));


	// Calculate SRF multiplier using the healpix class
	hp_kernel->calculateSRFMultiplier(MaxSphericalHarmonicsOrder, MicNumber, block_size / 2, micanglestheta, micanglesphi, kr_1_real, kr_1_imag);

	// Calculate the healpix multiplication
	hp_kernel->getmultiplicationresult(MaxSphericalHarmonicsOrder);
	multiplicationres = (float*)(hp_kernel->return_multiplicationresult_real());
	multiplicationres_imag = (float*)(hp_kernel->return_multiplicationresult_imag());

	max_hist = (float*)calloc(hist_size_forblocks, sizeof(float));

	prev_nofsource = 0;

}
// Sort the matrix
int* speechdetectionandseparation::SortInDescendingOrder(int NofPixels, int* inp_mt, float average)
{
	int* RENTHistogramIndex = (int*)calloc(NofPixels, sizeof(int));


	int i, j, min, temp, temp_ind;
	for (i = 0; i < NofPixels; i++) {
		RENTHistogramIndex[i] = i;
	}
	for (i = 0; i < NofPixels - 1; i++) {
		min = i;
		for (j = i + 1; j < NofPixels; j++)
			if (inp_mt[j] < inp_mt[min])
			{
				min = j;

			}
		temp_ind = RENTHistogramIndex[i];
		RENTHistogramIndex[i] = RENTHistogramIndex[min];
		RENTHistogramIndex[min] = temp_ind;

		temp = inp_mt[i];
		inp_mt[i] = inp_mt[min];
		inp_mt[min] = temp;
	}
	for (i = 0; i < NofPixels; i++) {

		if (inp_mt[i] <= (int)average)
		{
			inp_mt[i] = 0;

		}
	}
	// It is a matrix containing sorted index of array
	return RENTHistogramIndex;

}



complex_num speechdetectionandseparation::SpeechSeparation()
{
	cudaError_t cudaStatus;
	complex_num res;

	for (int it = 1; it < RENT_iteration; it++)
	{


		/*SRF is multiplied with the Legendre Kernels*/
/*Main aim is to calculate the maximum valued kernel*/
/* Input : dev_c,dev_c_im : SRF  [10][384][192]*/
/* Input : dev_kernel : Legendre kernels [384][192]*/
/* Output : dev_a, dev_a_im [10][384][192] */
			//Corrected

		cudaStatus = Device_DmultiplyWithCuda((float*)dev_a, (float*)dev_a_im, (float*)dev_c, (float*)dev_kernel, (float*)dev_c_im, NumberofFrequencybins, AudioBlockSize, NofFrequencyBins, FrequencybinAnalysisStart, healpix_samplingpoints, (float*)dev_kr_matrix);
		if (cudaStatus != cudaSuccess) {
			fprintf(stderr, "addWithCuda failed! 1");
			return res;
		}

		// Get the maximum valued Legendre Kernels for each time-frequency bin.
	/*Main aim is to get the histogram for the maximum valued kernel*/
	/* Input : dev_a, dev_a_im [10][192][192]  : Multiplication result*/
	/* Output : dev_histogram [10][192] : Integer valued histogram */
				//Corrected

		cudaStatus = Device_GetHistogram((float*)dev_a, (float*)dev_a_im, (float*)dev_histogram, NumberofFrequencybins, healpix_samplingpoints, AudioBlockSize);
		if (cudaStatus != cudaSuccess) {
			fprintf(stderr, "addWithCuda failed! 2");
			return res;
		}

		// By using the SRF and ResidualCalculator vector
	/*Main aim is to calculate the residual*/
	/* Input : dev_c, dev_c_im [10][192 *SRF_Plane][192] : SRF */
	/* Input : dev_histogram [10][192 *SRF_Plane] : Integer valued histogram*/
	/* Input : dev_res_kernel [192][192][192] : Residual calculator matrix for each legendre kernel*/
	/* Output : dev_a, dev_a_im [10][192 *SRF_Plane][192] : Residual SRF */


		cudaStatus = RescomplexmatrixMultiplicationKernel((float*)dev_a, (float*)dev_a_im, (float*)dev_c, (float*)dev_res_kernel, (float*)dev_c_im, (float*)dev_histogram, NumberofFrequencybins, healpix_samplingpoints, AudioBlockSize, (float*)dev_kr_matrix);
		if (cudaStatus != cudaSuccess) {
			fprintf(stderr, "addWithCuda failed! 3");
			return res;
		}

		// By using the SRF and ResidualCalculator vector
	/* Input : dev_c, dev_c_im [10][192][192] : SRF */
	/* Input : dev_histogram [10][192 * SRF_Plane] : Integer valued histogram*/
	/* Input : dev_pinv_kernel [192][192] : Magnitude calculator matrix for each legendre kernel*/
	/* Output : dev_magnitudes, dev_magnitudes_im [10][384] : Magnitude matrix */
		cudaStatus = MagcomplexmatrixMultiplicationKernel((float*)dev_magnitudes, (float*)dev_magnitudes_im, (float*)dev_c, (float*)dev_pinv_kernel, (float*)dev_c_im, (float*)dev_histogram, NumberofFrequencybins, healpix_samplingpoints, AudioBlockSize, (float*)dev_kr_matrix);
		if (cudaStatus != cudaSuccess) {
			fprintf(stderr, "addWithCuda failed! 4");
			return res;
		}

		cudaStatus = cudaMemcpy((c_hist + it * NumberofFrequencybins * AudioBlockSize), dev_histogram, sizeof(float) * NumberofFrequencybins * AudioBlockSize, cudaMemcpyDeviceToHost);
		if (cudaStatus != cudaSuccess) {
			fprintf(stderr, "cudaMemcpy failed! 5");
			return res;
		}

		cudaStatus = Swapmatrices2D((float*)(dev_magnitudes_final + it * NumberofFrequencybins * AudioBlockSize), (float*)(dev_magnitudes_im_final + it * NumberofFrequencybins * AudioBlockSize), (float*)dev_magnitudes, (float*)dev_magnitudes_im, NumberofFrequencybins, AudioBlockSize);
		if (cudaStatus != cudaSuccess) {
			fprintf(stderr, "addWithCuda failed! 6");
			return res;
		}

		cudaStatus = Swapmatrices((float*)dev_c, (float*)dev_c_im, (float*)dev_a, (float*)dev_a_im, NumberofFrequencybins, AudioBlockSize, healpix_samplingpoints);
		if (cudaStatus != cudaSuccess) {
			fprintf(stderr, "addWithCuda failed! 7");
			return res;
		}
	}

	// Copy Host to device with [NofIterations][NofBlocks][Nofreqencybins]
	// It contains max valued kernel pixel numbers
		cudaStatus = cudaMemcpy((float*)dev_general_histogram, (float*)c_hist, RENT_iteration * sizeof(float) * NumberofFrequencybins * AudioBlockSize, cudaMemcpyHostToDevice);
		if (cudaStatus != cudaSuccess) {
			fprintf(stderr, "cudaMemcpy failed 12!");
		}

		// Pixel numbers for each TF bin
		/* Input : dev_multiplication_kernel [192(fordoa)][192(forthepixel)] :  */
		/* Input : dev_magnitudes, dev_magnitudes_im [iterationnumber][NofBlocks][Noffrequency] : Magnitudes for each bin */
		/* Output : computedmagnitudes [10][getmultiplicationresult1024] : Separated source : For each source*/

		cudaStatus = OMPSeparatedSources((float*)dev_result, (float*)dev_result_im, (float*)dev_prevsep, (float*)dev_prevsep_im, (float*)dev_multiplication_kernel, (float*)dev_multiplication_kernel_imag, (float*)dev_magnitudes_final, (float*)dev_magnitudes_im_final, (float*)dev_general_histogram, AudioBlockSize, healpix_samplingpoints, NumberofFrequencybins, NofFrequencyBins, RENT_iteration, (float*)source_doa_hp, source_doas.size(), prev_nofsource,(float*)dev_pressure_level, (float*)dev_kr_matrix);
		if (cudaStatus != cudaSuccess) {
			fprintf(stderr, "addWithCuda failed! 8");
			return res;
		}

		/////////////////////////////////////////////////////////////////////////////////////

		
		cudaStatus = cudaMemcpy((float*)actual_result, (float*)dev_result, sizeof(float) * AudioBlockSize * NumberofFrequencybins * source_doas.size(), cudaMemcpyDeviceToHost);
		if (cudaStatus != cudaSuccess) {
			fprintf(stderr, "cudaMemcpy failed 13!");
			return res;
		}


		cudaStatus = cudaMemcpy((float*)actual_result_im, (float*)dev_result_im, sizeof(float) * AudioBlockSize * NumberofFrequencybins * source_doas.size(), cudaMemcpyDeviceToHost);
		if (cudaStatus != cudaSuccess) {
			fprintf(stderr, "cudaMemcpy failed 14!");
			return res;
		}

		prev_nofsource = source_doas.size();

		cudaStatus = cudaMemcpy((float*)dev_prevsep, (float*)dev_result, sizeof(float) * AudioBlockSize * NumberofFrequencybins * source_doas.size(), cudaMemcpyDeviceToDevice);
		if (cudaStatus != cudaSuccess) {
			fprintf(stderr, "cudaMemcpy failed 13!");
			return res;
		}

		cudaStatus = cudaMemcpy((float*)dev_prevsep_im, (float*)dev_result_im, sizeof(float) * AudioBlockSize * NumberofFrequencybins * source_doas.size(), cudaMemcpyDeviceToDevice);
		if (cudaStatus != cudaSuccess) {
			fprintf(stderr, "cudaMemcpy failed 14!");
			return res;
		}

		// Copy output vector from GPU buffer to host memory.
		//float* c_i = (float*)malloc(NumberofFrequencybins * AudioBlockSize * source_doas.size() * sizeof(float));

		//cudaStatus = cudaMemcpy(c_i, (float*)actual_result, NumberofFrequencybins * AudioBlockSize * source_doas.size() * sizeof(float), cudaMemcpyDeviceToHost);
		//if (cudaStatus != cudaSuccess) {
		//	fprintf(stderr, "cudaMemcpy failed!");
		//	return res;
		//}

	/*	printf("c--------------------------------\n");
		for (int k = 0; k < AudioBlockSize; k++) {

		for (int j = 0; j < NumberofFrequencybins; j++) {
			
			for (int i = 0; i < source_doas.size(); i++) {
				
					printf("(%d,%d,%d) = {%f},\n",
						i, k, j, actual_result_im[k * source_doas.size() * NofFrequencyBins + j * source_doas.size() + i]);
				}
			}
		}

		printf("\n");
		*/
		res.real = actual_result;
		res.imag = actual_result_im;
		//res.pressurelevel = c_i;
		return res;

}


// This function returns SRF calculated for the Block
complex_num speechdetectionandseparation::Return_BlockSRF()
{
	complex_num res;

	res.real = SRF_real;
	res.imag = SRF_imag;
	return res;
}
// Calculate the DOA of the sound sources and identify the sources. 
int speechdetectionandseparation::RENTComputation(float* stft_ch_real, float* stft_ch_imag)
{

	cudaError_t cudaStatus;
	float maxfreq = (float)sampling_rate * PI * 0.042 / 340.0;

	// Copy input vectors from host memory to GPU buffers.
	// [10][1024][32] input vector is STFT output 
	cudaStatus = cudaMemcpy((float*)dev_input, (float*)stft_ch_real, sizeof(float) * DD_InputDataFormat, cudaMemcpyHostToDevice);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMemcpy failed 15!");
	}
	// Copy input vectors from host memory to GPU buffers.
	cudaStatus = cudaMemcpy((float*)dev_input_im, (float*)stft_ch_imag, sizeof(float) * DD_InputDataFormat, cudaMemcpyHostToDevice);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMemcpy failed 16!");
	}

	//////////While calculate OMP in each bin up to K times
			// Here , we have plane vector that consists of the SRF obtained through the time-interval.
//  [NofAudioBlocks,NofSamplesinAudioBlock,NofMicrophones][NofMicrophones,NofLegendreSamples]
// Result will be [NofAudioBlocks,NofFrequencybins,NofLegendreSamples]
// SRF is calculated using this function 5 times
// We have on hand [NofAudioBlocks,NofFrequencybins,NofLegendreSamples]

				//  const int PlanematrixSize = (const int)PlaneDim * (*MSize) * (*MSize);
	cudaStatus = Device_DComplexmultiplyWithCuda((float*)dev_c, (float*)dev_c_im, (float*)dev_input, (float*)dev_b, (float*)dev_input_im, (float*)dev_b_im, nofmicrophones, NofFrequencyBins, NumberofFrequencybins, AudioBlockSize, FrequencybinAnalysisStart, healpix_samplingpoints, (float*)dev_kr_matrix);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "addWithCuda failed! 9");
		return 0;
	}

	/*For testing purposes, Kernels are inserted into the SRF*/
	/**************************************************************/
	int DD_SRF = AudioBlockSize * NumberofFrequencybins * healpix_samplingpoints;

/*			cudaStatus = cudaMemcpy((float*)dev_c, (float*)srf_magkernels, sizeof(float) * DD_SRF, cudaMemcpyHostToDevice);
					if (cudaStatus != cudaSuccess) {
						fprintf(stderr, "cudaMemcpy failed!");
					}
					// Copy input vectors from host memory to GPU buffers.
					cudaStatus = cudaMemcpy((float*)dev_c_im, (float*)srf_magkernels_img,  sizeof(float) * DD_SRF, cudaMemcpyHostToDevice);
					if (cudaStatus != cudaSuccess) {
						fprintf(stderr, "cudaMemcpy failed!");
					}
	/**************************************************************/


	// Copy input vectors from host memory to GPU buffers.
	cudaStatus = cudaMemcpy((float*)SRF_real, (float*)dev_c, sizeof(float) * DD_SRF, cudaMemcpyDeviceToHost);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMemcpy failed 16!");
	}

	cudaStatus = cudaMemcpy((float*)SRF_imag, (float*)dev_c_im, sizeof(float) * DD_SRF, cudaMemcpyDeviceToHost);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMemcpy failed 16!");
	}


	/*SRF is multiplied with the Legendre Kernels*/
/*Main aim is to calculate the maximum valued kernel*/
/* Input : dev_c,dev_c_im : SRF  [10][384][192]*/
/* Input : dev_kernel : Legendre kernels [192][192]*/
/* Output : dev_a, dev_a_im [10][384][192] */
			//Corrected

	cudaStatus = Device_DmultiplyWithCuda((float*)dev_a, (float*)dev_a_im, (float*)dev_c, (float*)dev_kernel, (float*)dev_c_im, NumberofFrequencybins, AudioBlockSize, NofFrequencyBins, FrequencybinAnalysisStart, healpix_samplingpoints, (float*)dev_kr_matrix);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "addWithCuda failed! 10");
		return 0;
	}

	// Get the maximum valued Legendre Kernels for each time-frequency bin.
/*Main aim is to get the histogram for the maximum valued kernel*/
/* Input : dev_a, dev_a_im [10][1024][192]  : Multiplication result*/
/* Output : dev_histogram [10][1024] : Integer valued histogram */
			//Corrected

	cudaStatus = Device_GetHistogram((float*)dev_a, (float*)dev_a_im, (float*)dev_histogram, NumberofFrequencybins, healpix_samplingpoints, AudioBlockSize);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "addWithCuda failed! 11");
		return 0;
	}

	// By using the SRF and ResidualCalculator vector
/*Main aim is to calculate the residual*/
/* Input : dev_c, dev_c_im [2][1024][192] : SRF */
/* Input : dev_histogram [2][1024] : Integer valued histogram*/
/* Input : dev_res_kernel [192][192][192] : Residual calculator matrix for each legendre kernel*/
/* Output : dev_a, dev_a_im [2][1024][192] : Residual SRF */


	cudaStatus = RescomplexmatrixMultiplicationKernel((float*)dev_a, (float*)dev_a_im, (float*)dev_c, (float*)dev_res_kernel, (float*)dev_c_im, (float*)dev_histogram, NumberofFrequencybins, healpix_samplingpoints, AudioBlockSize, (float*)dev_kr_matrix);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "addWithCuda failed! 12");
		return 0;
	}

	// By using the SRF and ResidualCalculator vector
/* Input : dev_c, dev_c_im [2][1024][192] : SRF */
/* Input : dev_histogram [2][1024] : Integer valued histogram*/
/* Input : dev_pinv_kernel [192][192] : Magnitude calculator matrix for each legendre kernel*/
/* Output : dev_magnitudes, dev_magnitudes_im [2][1024] : Magnitude matrix */
	cudaStatus = MagcomplexmatrixMultiplicationKernel((float*)dev_magnitudes, (float*)dev_magnitudes_im, (float*)dev_c, (float*)dev_pinv_kernel, (float*)dev_c_im, (float*)dev_histogram, NumberofFrequencybins, healpix_samplingpoints, AudioBlockSize,(float*)dev_kr_matrix);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "addWithCuda failed! 13");
		return 0;
	}

	cudaStatus = cudaMemcpy(c_hist , dev_histogram, sizeof(float) * NumberofFrequencybins * AudioBlockSize, cudaMemcpyDeviceToHost);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMemcpy failed 17!");
		return 0;
	}

	cudaStatus = Swapmatrices2D((float*)dev_magnitudes_final, (float*)dev_magnitudes_im_final, (float*)dev_magnitudes, (float*)dev_magnitudes_im, NumberofFrequencybins, AudioBlockSize);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "addWithCuda failed! 14");
		return 0;
	}

	// By using the SRF and ResidualCalculator vector
/* Input : dev_c, dev_c_im [2][1024][192] : SRF */
/* Input : dev_a, dev_a_im [2][1024][192] : Residual Error Vector */
/* Output : dev_RENT_histogram [2][1024] : RENT Histogram matrix for DOA estimation */

	cudaStatus = RENTHistogramKernel((float*)dev_c, (float*)dev_c_im, (float*)dev_a, (float*)dev_a_im, (float*)dev_histogram, (float*)dev_RENT_histogram, healpix_samplingpoints, AudioBlockSize, NumberofFrequencybins);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "addWithCuda failed! 15");
		return 0;
	}

	cudaStatus = cudaMemcpy(RENThist, (void *)dev_RENT_histogram, sizeof(float) * NumberofFrequencybins * AudioBlockSize, cudaMemcpyDeviceToHost);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMemcpy failed 18!");
		return 0;
	}

	cudaStatus = cudaMemcpy((void*)dev_RENT_histogram, max_hist, sizeof(float) * NumberofFrequencybins *  AudioBlockSize, cudaMemcpyHostToDevice);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMemcpy failed 19!");
		return 0;
	}

	cudaStatus = Swapmatrices((float*)dev_c, (float*)dev_c_im, (float*)dev_a, (float*)dev_a_im, NumberofFrequencybins, AudioBlockSize, healpix_samplingpoints);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "addWithCuda failed! 16");
		return 0;
	}


	float* Histogram = RENThist;
	float* MaxValuedHistogram = c_hist;
	int NofPixels = healpix_samplingpoints;
	int PlaneDim = AudioBlockSize;
	int NofFreq = NumberofFrequencybins;

	vector<int> neighbouringpixels = vector<int>();
	int* RENTHistogram = (int*)calloc(NofPixels, sizeof(int));
	float average = (float)NofPixels;
	for (int plj = 0; plj < PlaneDim; plj++)
	{
		for (int klj = 84; klj < RENTanalysis; klj++)
		{
			if (Histogram[plj * NofFreq + klj] < RENTThreshold)
			{
				int index = MaxValuedHistogram[plj * NofFreq + klj];
				RENTHistogram[index]++;
				average = average + 1.0;
			}
		}
	}

	float nofsources = (float)0.0;
	for (int klj = 0; klj < NofPixels; klj++)
	{
		if (RENTHistogram[klj] > 0)
			nofsources = nofsources + 1.0;
	}


	average = average / (float)NofPixels;



	int* sortedarray = SortInDescendingOrder(NofPixels, RENTHistogram, average);
	float angle;
	bool insertangle;

	int* neighbouring = (int*)calloc(8, sizeof(int));
	for (int plj = NofPixels - 1; plj >= 0; plj--)
	{
		//printf("(Source : %d) = {%d,%d}", plj, RENTHistogram[plj], sortedarray[plj]);
		if (RENTHistogram[plj] > (int)0)
		{
			DOA_Data m_doa = DOA_Data();
			m_doa.distance = 1.0;
			m_doa.hpnumber = sortedarray[plj];
			m_doa.firstin = true;
			m_doa.activeframe = 1;
			m_doa.uuid = boost::uuids::to_string(boost::uuids::random_generator()());
			hp_kernel->gethealpixcoordinates(sortedarray[plj], m_doa.theta, m_doa.phi);
			hp_kernel->ConvertSphericalToCartesian(m_doa.distance, m_doa.theta, m_doa.phi, m_doa.x, m_doa.y, m_doa.z);
			insertangle = true;


			for (int sid = 0; sid < source_doas.size(); sid++)
			{

				if (source_doas[sid].hpnumber == m_doa.hpnumber)
				{
					insertangle = false;
					break;
				}

				if (count(neighbouringpixels.begin(), neighbouringpixels.end(), m_doa.hpnumber))
				{
					insertangle = false;
					neighbouring = hp_kernel->gethealpixneighbours(m_doa.hpnumber);
					for (int klmn = 0; klmn < 8; klmn++)
					{
						neighbouringpixels.push_back(neighbouring[klmn]);
					}
					break;

				}

				angle = hp_kernel->FindAngleBtw(m_doa.x, m_doa.y, m_doa.z, m_doa.distance, source_doas[sid].x, source_doas[sid].y, source_doas[sid].z, source_doas[sid].distance);
				//  printf("(Source : %d) = {%f}", plj, angle);
				if (abs(angle) < AngleDiff)
				{
					insertangle = false;
					neighbouring = hp_kernel->gethealpixneighbours(m_doa.hpnumber);
					neighbouringpixels.push_back(m_doa.hpnumber);
					for (int klmn = 0; klmn < 8; klmn++)
					{
						neighbouringpixels.push_back(neighbouring[klmn]);
					}

					/*source_doas[sid].x -= m_doa.x;
					source_doas[sid].y -= m_doa.y;
					source_doas[sid].z -= m_doa.z;
					hpkernel->ConvertCartesionToSpherical(m_doa.distance, m_doa.theta, m_doa.phi, m_doa.x, m_doa.y, m_doa.z);
					*/
					source_doas[sid].activeframe++;
					break;
				}
			}
			if (insertangle || source_doas.size() == 0)
			{
				source_doas.push_back(m_doa);
				neighbouring = hp_kernel->gethealpixneighbours(m_doa.hpnumber);
				neighbouringpixels.push_back(m_doa.hpnumber);
				for (int klmn = 0; klmn < 8; klmn++)
				{
					neighbouringpixels.push_back(neighbouring[klmn]);
				}

				insertangle = true;
			}
		}
	}

	/* This matrix is used for result calculations*/
/***********************************************************/
	cudaStatus = cudaMalloc((void**)&dev_result, DD_ResultDataFormat * source_doas.size() * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc failed!");
	}

	cudaStatus = cudaMalloc((void**)&dev_result_im, DD_ResultDataFormat * source_doas.size() * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc failed!");
	}

	cudaStatus = cudaMalloc((void**)&dev_prevsep, DD_ResultDataFormat * source_doas.size() * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc failed!");
	}

	cudaStatus = cudaMalloc((void**)&dev_prevsep_im, DD_ResultDataFormat * source_doas.size() * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc failed!");
	}

	cudaStatus = cudaMalloc((void**)&dev_pressure_level, source_doas.size() * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc failed!");

	cudaMemset((void*)dev_pressure_level, 0, source_doas.size() * sizeof(float));

	}

	/***********************************************************/


	actual_result = (float*)calloc(DD_ResultDataFormat * source_doas.size(), sizeof(float));

	actual_result_im = (float*)calloc(DD_ResultDataFormat * source_doas.size(), sizeof(float));

	host_source_doa_hp = (float*)calloc(MaxNumberofSeparatedObjects, sizeof(float));

	printf("\n");
	for (int sid = 0; sid < source_doas.size(); sid++)
	{
		// printf("(Source :  %d) = {%d}", sid, neighbouringpixels[sid]);
		printf("(Source : %d, %d) = {%f,%f}", sid, source_doas[sid].hpnumber, 180 * source_doas[sid].theta / 3.14, (180 * source_doas[sid].phi / 3.14));
		//hpkernel->create_vonmises(source_doas[sid].hpnumber);
		printf("\n");

		host_source_doa_hp[sid] = (float)source_doas[sid].hpnumber;
	}

	cudaStatus = cudaMemcpy((void*)source_doa_hp, host_source_doa_hp, MaxNumberofSeparatedObjects * sizeof(float), cudaMemcpyHostToDevice);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMemcpy failed 20!");
	}
	BlockIndex++;


	return source_doas.size();

}

void* speechdetectionandseparation::initialization(int windowsize)
{
	this->WindowSize = windowsize;
	//3D operation where 
	const int PlaneDim = AudioBlockSize;
	NofFrequencyBins = (windowsize / 2);
	const int arraySizeH = NofFrequencyBins;
	const int arraySizeW = nofmicrophones;
	const int arraySize_H = nofmicrophones;
	const int arraySize_W = healpix_samplingpoints;

	const int FrequencystartAnalysis = FrequencybinAnalysisStart;
	const int Windowsize = windowsize;

	const int Maxnumberofobjects = MaxNumberofSeparatedObjects;
	const int FrequencybinAnalysis = NumberofFrequencybins;

	PartitionedSamples = AudioBlockSize * NofFrequencyBins;
	RENT_iteration = RENTIIteration;
	float ratio = RENTAnalysisRange;

	RENTanalysis = (int)((float)NofFrequencyBins * ratio);
// For Each iteration AudioBlockSize * NumberofFrequencybinAnalysis
	int DD_RENTHistogram = PlaneDim * FrequencybinAnalysis;
// For RENT operation IterationNumber * AudioBlockSize * NumberofFrequencybinAnalysis
	int DD_IterationRENTHistogram = RENT_iteration * PlaneDim * FrequencybinAnalysis;
// For RENT operation SRF is calculated in healpix dimensions AudioBlockSize * NumberofFrequencybinAnalysis * HP_SamplingPoints
	int DD_SRF = PlaneDim * FrequencybinAnalysis * healpix_samplingpoints;
// For RENT operation Residual SRF is calculated in healpix dimensions AudioBlockSize * NumberofFrequencybinAnalysis * HP_SamplingPoints
	int DD_Res_SRF = PlaneDim * FrequencybinAnalysis * healpix_samplingpoints;
// Legendre kernels calculated in HP_SamplingPoints * HP_SamplingPoints
	int DD_LegendreKernels = healpix_samplingpoints * healpix_samplingpoints;
	int DD_LegendreKernels_hp = MaxSphericalHarmonicsOrder * healpix_samplingpoints * healpix_samplingpoints;

	int DD_RES_LegendreKernels = MaxSphericalHarmonicsOrder * healpix_samplingpoints * healpix_samplingpoints * healpix_samplingpoints;
	int DD_RES_LegendreKernels_hp = healpix_samplingpoints * healpix_samplingpoints * healpix_samplingpoints;

	int DD_PIV_LegendreKernels = MaxSphericalHarmonicsOrder * healpix_samplingpoints * healpix_samplingpoints;
// Calculated Legendre Kernel magnitudes
	int DD_MagnitudesForKernels = PlaneDim * FrequencybinAnalysis * RENT_iteration;
	int DD_MagnitudesForKernelsInBlock = PlaneDim * FrequencybinAnalysis;
	int DD_IterationMagnitudeForKernels = RENT_iteration * PlaneDim * FrequencybinAnalysis * healpix_samplingpoints;
	int DD_VonMisesForKernels = healpix_samplingpoints * healpix_samplingpoints;
	int DD_KrValueforSHD = MaxSphericalHarmonicsOrder* arraySizeW * healpix_samplingpoints;

	DD_InputDataFormat = arraySizeW * PlaneDim * NofFrequencyBins;
	DD_ResultDataFormat = PlaneDim * NofFrequencyBins;

	cudaError_t cudaStatus;


	c_hist = (float*)calloc(RENT_iteration * DD_MagnitudesForKernelsInBlock, sizeof(float));
	c_hist_im = (float*)calloc(RENT_iteration * DD_MagnitudesForKernelsInBlock, sizeof(float));

	kr_matrix = (float*)calloc(NofFrequencyBins, sizeof(float));
	float krfreq;
	float kr;

	for (int klx = 0; klx < NofFrequencyBins; klx++)
	{
		krfreq = (float)sampling_rate * PI * 0.042 * (float)klx / (170.0 * (float)NofFrequencyBins);

		if (krfreq < 1)
			kr = 0.0;
		else if (krfreq < 2)
			kr = 1.0;
		else if (krfreq < 3)
			kr = 2.0;
		else
			kr = 3.0;

		*(kr_matrix + klx) = (float)kr;
	}

	BlockIndex = 0;

	// Create a hamming window of appropriate length
	//hammingwindow(block_size);

	// Choose which GPU to run on, change this on a multi-GPU system.
//cudaStatus = cudaSetDevice(0);
//	if (cudaStatus != cudaSuccess) {
//		fprintf(stderr, "cudaSetDevice failed!  Do you have a CUDA-capable GPU installed?");
//	}

/* This matrix is taken as input from the microphone array */
/***********************************************************/
	cudaStatus = cudaMalloc((void**)&dev_input, DD_InputDataFormat * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc microphone array stft real matrices!");
	}
	cudaStatus = cudaMalloc((void**)&dev_input_im, DD_InputDataFormat * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc microphone array stft imag matrices!");
	}

	cudaMemset((void*)dev_input, 0, DD_InputDataFormat * sizeof(float));
	cudaMemset((void*)dev_input_im, 0, DD_InputDataFormat * sizeof(float));
	/***********************************************************/

/* Kernel matrix composition */
/***********************************************************/
	cudaStatus = cudaMalloc((void**)&dev_kernel, DD_LegendreKernels_hp * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc kernel matrix failed!");
	}
/***********************************************************/
/***********************************************************/
	cudaStatus = cudaMalloc((void**)&dev_kr_matrix, NofFrequencyBins * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc bnkra selection matrix failed!");
	}
	/***********************************************************/

/* This matrix is taken as residual energy calculator matrix*/
/***********************************************************/
	cudaStatus = cudaMalloc((void**)&dev_res_kernel, DD_RES_LegendreKernels * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc residual energy calculator matrix failed!");
	}
	/***********************************************************/
	/***********************************************************/

/* This matrix is taken as pinv calculator matrix*/
/***********************************************************/
	cudaStatus = cudaMalloc((void**)&dev_pinv_kernel, DD_PIV_LegendreKernels * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc PINV Matrix failed!");
	}
	/***********************************************************/

	/* This matrix is used as SRF real and imaginery*/
/***********************************************************/
	cudaStatus = cudaMalloc((void**)&dev_c, DD_SRF * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc SRF Matrix real failed!");
	}

	// Allocate GPU buffers for three vectors (two input, one output)    .
	cudaStatus = cudaMalloc((void**)&dev_c_im, DD_SRF * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc SRF Matrix imag failed!");
	}

	cudaMemset((void*)dev_c, 0, DD_SRF * sizeof(float));
	cudaMemset((void*)dev_c_im, 0, DD_SRF * sizeof(float));

// Host SRF matrix malloc with the following sizes
	SRF_real = (float*)malloc(DD_SRF * sizeof(float));
	SRF_imag = (float*)malloc(DD_SRF * sizeof(float));
	/***********************************************************/

	/* This matrix is used as SRF real and imaginery same decomposition*/
/***********************************************************/
	cudaStatus = cudaMalloc((void**)&dev_a, DD_SRF * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc failed!");
	}
	cudaStatus = cudaMalloc((void**)&dev_a_im, DD_SRF * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc failed!");
	}

	cudaMemset((void*)dev_a, 0, DD_SRF * sizeof(float));
	cudaMemset((void*)dev_a_im, 0, DD_SRF * sizeof(float));

	/***********************************************************/
	/***********************************************************/

	/* This matrix is used as SRF real and imaginery same decomposition*/
	/***********************************************************/
	cudaStatus = cudaMalloc((void**)&dev_b, DD_KrValueforSHD * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc failed!");
	}
	cudaStatus = cudaMalloc((void**)&dev_b_im, DD_KrValueforSHD * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc failed!");
	}
	cudaMemset((void*)dev_b, 0, DD_KrValueforSHD * sizeof(float));
	cudaMemset((void*)dev_b_im, 0, DD_KrValueforSHD * sizeof(float));


	cudaStatus = cudaMalloc((void**)&source_doa_hp, MaxNumberofSeparatedObjects * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc failed!");
	}

	cudaMemset((void*)source_doa_hp, 0, MaxNumberofSeparatedObjects * sizeof(float));

	/***********************************************************/
	/* These matrices are used to create histogram for RENT and internal usage*/
	/***********************************************************/

	cudaStatus = cudaMalloc((void**)&dev_histogram, DD_RENTHistogram * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc failed!");
	}

	cudaStatus = cudaMalloc((void**)&dev_RENT_histogram, DD_RENTHistogram * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc failed!");
	}

	/***********************************************************/
	/***********************************************************/

	/* This matrix is used to compute the magnitude for the detected kernel*/
	/***********************************************************/

	cudaStatus = cudaMalloc((void**)&dev_magnitudes, DD_MagnitudesForKernelsInBlock * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc failed!");
	}

	cudaStatus = cudaMalloc((void**)&dev_magnitudes_im, DD_MagnitudesForKernelsInBlock * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc failed!");
	}

	/***********************************************************/
	/***********************************************************/

	/* This matrix is used to store the magnitudes for iterations*/
	/***********************************************************/

	cudaStatus = cudaMalloc((void**)&dev_magnitudes_final, DD_MagnitudesForKernels * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc failed!");
	}

	cudaStatus = cudaMalloc((void**)&dev_magnitudes_im_final, DD_MagnitudesForKernels * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc failed!");
	}

	/***********************************************************/
	/***********************************************************/

	/* This matrix is used to store the histogram obtained in each iteration*/
	/***********************************************************/
	cudaStatus = cudaMalloc((void**)&dev_general_histogram, DD_IterationRENTHistogram * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc failed!");
	}

	/***********************************************************/
	/***********************************************************/

	/* This matrix is used for vonmises calculations*/
	/***********************************************************/

	cudaStatus = cudaMalloc((void**)&dev_multiplication_kernel, MaxSphericalHarmonicsOrder * DD_VonMisesForKernels * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc failed!");
	}
	cudaStatus = cudaMemcpy((void*)dev_multiplication_kernel, multiplicationres, MaxSphericalHarmonicsOrder * DD_VonMisesForKernels * sizeof(float), cudaMemcpyHostToDevice);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMemcpy failed 21!");
	}

	cudaStatus = cudaMalloc((void**)&dev_multiplication_kernel_imag, MaxSphericalHarmonicsOrder * DD_VonMisesForKernels * sizeof(float));
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMalloc failed!");
	}
	cudaStatus = cudaMemcpy((void*)dev_multiplication_kernel_imag, multiplicationres_imag, MaxSphericalHarmonicsOrder * DD_VonMisesForKernels * sizeof(float), cudaMemcpyHostToDevice);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMemcpy failed 21!");
	}

	/***********************************************************/
	/***********************************************************/

	for (int idx = 0; idx < MaxSphericalHarmonicsOrder; idx++)
	{
		// Calculate the legendre kernels with the specified spherical harmonics order
		// The format is in N*[Pixelsize*Pixelsize] where N is the maximum spherical harmonics order
		// The imaginary parts of the dictionary kernels
		magkernels = hp_kernel->calculatelegendrekernels(idx+1);
		hp_kernel->calculateorthogonalLK();

		/* Copy magnitude kernels to kernel matrix*/
	/***********************************************************/
		cudaStatus = cudaMemcpy((void*)(dev_kernel + idx * DD_LegendreKernels), magkernels, DD_LegendreKernels * sizeof(float), cudaMemcpyHostToDevice);
		if (cudaStatus != cudaSuccess) {
			fprintf(stderr, "cudaMemcpy failed!");
		}
		/***********************************************************/
		/***********************************************************/


		float* fl;
		/* Calculate pinv and residual energy matrix values*/
		/***********************************************************/
		for (int kl = 0; kl < healpix_samplingpoints; kl++)
		{

			fl = hp_kernel->getpinvlegendrekernels(kl);

			cudaStatus = cudaMemcpy((void*)(dev_res_kernel  + idx * DD_RES_LegendreKernels_hp + kl * DD_LegendreKernels), fl, DD_LegendreKernels * sizeof(float), cudaMemcpyHostToDevice);
			if (cudaStatus != cudaSuccess) {
				fprintf(stderr, "cudaMemcpy failed 22!");
			}

			fl = hp_kernel->getpinvkernels(kl);

			cudaStatus = cudaMemcpy((void*)(dev_pinv_kernel + idx * DD_LegendreKernels + kl * healpix_samplingpoints), fl, healpix_samplingpoints * sizeof(float), cudaMemcpyHostToDevice);
			if (cudaStatus != cudaSuccess) {
				fprintf(stderr, "cudaMemcpy failed!");
			}

		}
	}
	/***********************************************************/
	/***********************************************************/

	/* Create histogram matrices*/
	/***********************************************************/
 	cudaStatus = cudaMemcpy((void*)dev_histogram, max_hist, DD_RENTHistogram * sizeof(float), cudaMemcpyHostToDevice);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMemcpy failed!");
	}

	cudaStatus = cudaMemcpy((void*)dev_RENT_histogram, max_hist, DD_RENTHistogram * sizeof(float), cudaMemcpyHostToDevice);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMemcpy failed!");
	}

	cudaStatus = cudaMemcpy((void*)dev_kr_matrix, kr_matrix, NofFrequencyBins * sizeof(float), cudaMemcpyHostToDevice);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMemcpy failed!");
	}
	/***********************************************************/
	/***********************************************************/

	/* Copy frequency smoothing functions*/
	/***********************************************************/
	cudaStatus = cudaMemcpy((void*)dev_b, kr_1_real, DD_KrValueforSHD * sizeof(float), cudaMemcpyHostToDevice);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMemcpy failed!");
	}

	cudaStatus = cudaMemcpy((void*)dev_b_im, kr_1_imag, DD_KrValueforSHD * sizeof(float), cudaMemcpyHostToDevice);
	if (cudaStatus != cudaSuccess) {
		fprintf(stderr, "cudaMemcpy failed!");
	}
	/***********************************************************/
	/***********************************************************/


	RENThist = (float*)malloc(AudioBlockSize * NumberofFrequencybins * sizeof(float));

	/*************************************************TESTING FROM SRF**********************************************************/
/*	srf_magkernels = (float*)calloc(AudioBlockSize * FrequencybinAnalysis * healpix_samplingpoints, sizeof(float));
	srf_magkernels_img = (float*)calloc(AudioBlockSize * FrequencybinAnalysis * healpix_samplingpoints, sizeof(float));

	float* srf_total = (float*)calloc(healpix_samplingpoints, sizeof(float));

	for (int jkl = 0; jkl < healpix_samplingpoints; jkl++)
	{
		srf_total[jkl] = 0.05 * magkernels[60 * healpix_samplingpoints + jkl]; // +0.5 * magkernels[75 * arraysize + jkl] + 1.0 * magkernels[14 * arraysize + jkl];
		//printf("(%d) = {%f},",jkl, magkernels[60 * healpix_samplingpoints + jkl]);
	}

	for (int klm = 0; klm < AudioBlockSize; klm++)
	{
		for (int ilm = 0; ilm < FrequencybinAnalysis; ilm++)
		{
			if(ilm == 128)
				memcpy(srf_magkernels + klm * healpix_samplingpoints * FrequencybinAnalysis + ilm * healpix_samplingpoints, srf_total, sizeof(float) * healpix_samplingpoints);
		}

	}

	for (int jkl = 0; jkl < healpix_samplingpoints; jkl++)
	{
		srf_total[jkl] = 0.05 * magkernels[102 * healpix_samplingpoints + jkl];
		//printf("(%d) = {%f},",jkl, magkernels[105 * arraysize + jkl]);
	}

	for (int klm = 0; klm < AudioBlockSize; klm++)
	{
		for (int ilm = 0; ilm < FrequencybinAnalysis; ilm++)
		{
			if (ilm == 256)
				memcpy(srf_magkernels + klm * healpix_samplingpoints * FrequencybinAnalysis + ilm * healpix_samplingpoints, srf_total, sizeof(float) * healpix_samplingpoints);
		}

	}
*/

	/*************************************************TESTING FROM SRF**********************************************************/

	return 0;


}

json speechdetectionandseparation::audioscenegeometry_out()
{

	json audioscenegeometry_out_file;
	audioscenegeometry_out_file["$schema"] = "Audio scene geometry";
	audioscenegeometry_out_file["title"] = "Audio scene geometry";
	audioscenegeometry_out_file["BlockIndex"] = BlockIndex;
	audioscenegeometry_out_file["BlockStart"] = 1;
	audioscenegeometry_out_file["BlockEnd"] = 1;
	audioscenegeometry_out_file["SpeechCount"] = source_doas.size();
	audioscenegeometry_out_file["SourceDetectionMask"] = 1;
	for(int idl= 0; idl < source_doas.size();idl++)
		audioscenegeometry_out_file["SpeechList"][idl] = { {"SpeechID" ,  source_doas[idl].uuid}, {"ChannelID" , idl }, {"AzimuthDirection" ,source_doas[idl].phi}, {"ElevationDirection" , source_doas[idl].theta}, {"DistanceFlag" , false}, {"Distance" , 0.0 } };

	return audioscenegeometry_out_file;
}