#include "../src/lib/core.hpp"

#include <gtest/gtest.h>
#include <filesystem>

#include <opencv2/core/core.hpp>

using namespace videoanalyser;
using namespace cv;

TEST(Core, Frame) {
    core::Frame f = core::Frame(cv::imread("tests/images/test_scene_01.jpg"));
    EXPECT_EQ(f.size().width, 720);
    EXPECT_EQ(f.size().height, 576);
    EXPECT_EQ(f.channels(), 3);
}

TEST(Core, FrameClone) {
    core::Frame f = core::Frame(cv::imread("tests/images/test_scene_01.jpg"));
    core::Frame f2 = f.clone();
    EXPECT_EQ(f.size().width, f2.size().width);
    EXPECT_EQ(f.size().height, f2.size().height);
}

TEST(Core, FrameDownsample) {
    core::Frame f = core::Frame(cv::imread("tests/images/test_scene_01.jpg"));
    core::Frame f2 = f.clone();
    f2.downsample(2);
    EXPECT_EQ(f.size().width / 2, f2.size().width);
    EXPECT_EQ(f.size().height / 2, f2.size().height);
}

TEST(Core, FrameConvertColor) {
    core::Frame f = core::Frame(cv::imread("tests/images/test_scene_01.jpg"));
    core::Frame f2 = f.clone();
    f2.convert_color(cv::COLOR_BGR2GRAY);
    EXPECT_EQ(f.size().width, f2.size().width);
    EXPECT_EQ(f.size().height, f2.size().height);
    EXPECT_EQ(f2.channels(), 1);
}

TEST(Core, FrameCrop) {
    core::Frame f = core::Frame(cv::imread("tests/images/test_scene_01.jpg"));
    core::Frame f2 = f.clone();
    f2.crop(cv::Size(100, 100), cv::Point2f(100, 100));
    EXPECT_EQ(f2.size().width, 100);
    EXPECT_EQ(f2.size().height, 100);
}

TEST(Core, FrameWarp) {
    core::Frame f = core::Frame(cv::imread("tests/images/test_scene_01.jpg"));
    core::Frame f2 = f.clone();
    cv::Mat rotationMatrix = cv::getRotationMatrix2D(cv::Point2f(f.size().width / 2, f.size().height / 2), 45, 1);
    f2.warp(rotationMatrix);
    EXPECT_EQ(f2.size().width, f.size().width);
    EXPECT_EQ(f2.size().height, f.size().height);
}

TEST(Core, FrameDeinterlace) {
    core::Frame f = core::Frame(cv::imread("tests/images/test_scene_01.jpg"));
    std::pair<core::Frame, core::Frame> f2 = f.deinterlace();
    EXPECT_EQ(f2.first.size().width, f2.second.size().width);
    EXPECT_EQ(f2.first.size().height, f2.second.size().height);
    EXPECT_EQ(f2.first.size().width, f.size().width);
    EXPECT_EQ(f2.first.size().height, f.size().height / 2);
}
