/**
 * @file detection.hpp
 * @author Matteo Spanio (dev2@audioinnova.com)
 * @brief This file contains the functions used to detect the objects in a scene.
 * @version 1.2
 * @date 2023-06-04
 *
 * @copyright Copyright (c) 2023
 *
 */

#ifndef VIDEOANALYSER_DETECTION_H
#define VIDEOANALYSER_DETECTION_H

#include <opencv2/calib3d.hpp>
#include <opencv2/core/core.hpp>
#include <opencv2/features2d.hpp>
#include <opencv2/imgcodecs.hpp>
#include <opencv2/imgproc.hpp>
#include <opencv2/xfeatures2d.hpp>
#include <variant>
#include <vector>

#include "core.hpp"

namespace videoanalyser {
namespace detection {
enum class ElementType { TAPE, CAPSTAN, READING_HEAD };
enum class Algorithm { GHT, SURF };
/**
 * @struct DetectionThreshold
 * @brief Struct containing the threshold values used to detect a shape.
 */
struct DetectionThreshold {
    float percentual; /**< The minimum percentage of different pixels for
                        considering the current frame under the ROI as a
                        potential Irregularity */
    int angle;        /**< The angle votes threshold for the detection of the object */
    int scale;        /**< The scale votes threshold for the detection of the object */
    int pos;          /**< The position votes threshold for the detection of the object
                       */
};

/**
 * @typedef Roi
 * @brief The region of interest of a scene. It is a pair containing the
 * coordinates of the top-left and bottom-right corners of the ROI and the
 * type of the object in the ROI.
 */
using Roi = cv::RotatedRect;
/**
 * @struct SceneElement
 * @brief A scene element is an object that can be detected in a scene, such as a
 * tape or a capstan.
 *
 */
struct SceneElement {
    ElementType type;             /**< The type of the object */
    int min_dist;                 /**< The minimum distance between the centers of the detected
                                   objects for the detection of the reading head */
    DetectionThreshold threshold; /**<  the threshold values used to detect the object */
};
/**
 * @fn Result<Roi> find_roi(core::Frame image, Algorithm algorithm, SceneElement element_to_find)
 * @brief Looks for a shape in a frame.
 *
 * Find a shape in a frame using the specified algorithm.
 *
 * @param image The frame in which the object will be searched.
 * @param algorithm The algorithm to use for the detection.
 * @param element_to_find The object to find.
 * @return Result<Roi<T>> A Result object containing the ROI of the object if found, otherwise an error.
 */
Result<Roi> find_roi(core::Frame image, Algorithm algorithm, SceneElement element_to_find);
}  // namespace detection
}  // namespace videoanalyser
#endif  // VIDEOANALYSER_DETECTION_H