/**
 * @file core.hpp
 * @author Matteo Spanio (dev2@audioinnova.com)
 * @brief This file contains the core functionalities of the project.
 * @version 1.2
 * @date 2023-06-03
 *
 * @copyright Copyright (c) 2023
 *
 */

#ifndef CORE_H
#define CORE_H
#include <opencv2/calib3d.hpp>
#include <opencv2/core/core.hpp>
#include <opencv2/features2d.hpp>
#include <opencv2/imgcodecs.hpp>
#include <opencv2/imgproc.hpp>
#include <opencv2/xfeatures2d.hpp>
#include <optional>
#include <string>
#include <tuple>
#include <variant>
#include <vector>

namespace videoanalyser {
using Error = std::string;
template <typename T>
using Result = std::variant<T, Error>;
namespace core {
/**
 * @class Frame
 * @brief Class that extends the OpenCV Mat class, adding some useful methods
 * frequently used in the project.
 *
 */
class Frame : public cv::Mat {
   public:
    Frame();
    Frame(const cv::Mat& m);
    Frame(const Frame& f);
    Frame& operator=(const cv::Mat& m);
    Frame& operator=(const Frame& f);
    Frame clone() const;
    /**
     * @brief Downsample the image by a given factor.
     *
     * @param factor The factor by which the image will be downsampled.
     * @return Frame& The downsampled image.
     */
    Frame& downsample(int factor);
    /**
     * @brief Convert the image to a given color space.
     *
     * @param code The code of the color space to which the image will be
     * converted.
     * @return Frame& The converted image.
     */
    Frame& convert_color(int code);
    /**
     * @brief Compute the number of different pixels between two frames.
     *
     * @param f The frame to compare with.
     * @return A black and white frame, where black pixels represent a
     * difference, while white pixels represent an equality.
     */
    Frame difference(Frame& f);
    /**
     * @brief Crop the image to a given size, centered in a given point.
     *
     * @param rect_size The size of the cropped image.
     * @param center The center of the cropped image.
     * @return Frame& The cropped image.
     */
    Frame& crop(cv::Size rect_size, cv::Point2f center);
    /**
     * @brief Warp the image using a given rotation matrix.
     *
     * @param rotationMatrix The rotation matrix used to warp the image.
     * @return Frame& The warped image.
     */
    Frame& warp(cv::Mat rotationMatrix);
    /**
     * @brief Deinterlace the image, returning two images, one containing the
     * odd lines and the other containing the even lines.
     *
     * @return std::pair<Frame, Frame> The two images containing the odd and
     * even lines.
     */
    std::pair<Frame, Frame> deinterlace() const;
};
}  // namespace core
}  // namespace videoanalyser
#endif  // CORE_H