#include <filesystem>
#include <opencv2/core/core.hpp>
#include <opencv2/highgui/highgui.hpp>
#include <opencv2/imgcodecs.hpp>
#include <opencv2/imgproc.hpp>

using namespace cv;
using namespace std;
namespace fs = std::filesystem;

/**
 * @brief Namespace containing a set of utility functions used in the project.
 * The functions are mainly used to perform operations on images.
 *
 */
namespace utility {

/**
 * @class Frame
 * @brief Class that extends the OpenCV Mat class, adding some useful methods
 * frequently used in the project.
 *
 */
class Frame : public Mat {
   public:
    Frame();
    Frame(const Mat& m);
    Frame(const Frame& f);
    Frame& operator=(const Mat& m);
    Frame& operator=(const Frame& f);
    Frame clone() const;
    /**
     * @brief Downsample the image by a given factor.
     *
     * @param factor The factor by which the image will be downsampled.
     * @return Frame& The downsampled image.
     */
    Frame& downsample(int factor);
    /**
     * @brief Convert the image to a given color space.
     *
     * @param code The code of the color space to which the image will be
     * converted.
     * @return Frame& The converted image.
     */
    Frame& convertColor(int code);
    Frame difference(Frame& f);
    /**
     * @brief Crop the image to a given size, centered in a given point.
     *
     * @param rect_size The size of the cropped image.
     * @param center The center of the cropped image.
     * @return Frame& The cropped image.
     */
    Frame& crop(Size rect_size, Point2f center);
    /**
     * @brief Warp the image using a given rotation matrix.
     *
     * @param rotationMatrix The rotation matrix used to warp the image.
     * @return Frame& The warped image.
     */
    Frame& warp(cv::Mat rotationMatrix);
    /**
     * @brief Deinterlace the image, returning two images, one containing the
     * odd lines and the other containing the even lines.
     *
     * @return std::pair<Frame, Frame> The two images containing the odd and
     * even lines.
     */
    std::pair<Frame, Frame> deinterlace() const;
};

/**
 * @fn void detectShape(Ptr<GeneralizedHoughGuil> alg, Mat templateShape, int
 * posThresh, vector<Vec4f> &positivePositions, Mat &positiveVotes,
 * vector<Vec4f> &negativePositions, Mat &negativeVotes, Mat processingArea)
 * @brief Detects a given shape in an image, using a the OpenCV algorithm
 * GeneralizedHoughGuil.
 *
 * @param[in] alg the algorithm instance;
 * @param[in] templateShape the shape to detect;
 * @param[in] posThresh the position votes threshold;
 * @param[out] positivePositions vector representing the position assigned to
 * each found rectangle for positive angles;
 * @param[out] positiveVotes vector representing the vote assigned to each found
 * rectangle for positive angles;
 * @param[out] negativePositions vector representing the position assigned to
 * each found rectangle for negative angles;
 * @param[out] negativeVotes vector representing the vote assigned to each found
 * rectangle for negative angles;
 * @param[in] processingArea the image to be processed.
 */
void detectShape(Ptr<GeneralizedHoughGuil> alg, Mat templateShape, int posThresh, vector<Vec4f>& positivePositions,
                 Mat& positiveVotes, vector<Vec4f>& negativePositions, Mat& negativeVotes, Mat processingArea);

/**
 * @fn RotatedRect drawShapes(Mat frame, Vec4f &positions, Scalar color, int
 * width, int height, int offsetX, int offsetY, float processingScale)
 * @brief Draw rectangles on an image.
 *
 * @param frame Frame on which the rectangles will be drawn;
 * @param positions The position of the rectangle;
 * @param color The color of the rectangle;
 * @param width The width of the rectangle;
 * @param height The height of the rectangle;
 * @param offsetX X offset on the position of the rectangle;
 * @param offsetY Y offset on the position of the rectangle;
 * @param processingScale Scaling factor, useful for downsizing.
 * @return RotatedRect Object representing the drawn rectangle.
 */
RotatedRect drawShapes(Mat frame, Vec4f& positions, Scalar color, int width, int height, int offsetX, int offsetY,
                       float processingScale);

/**
 * @fn void separateFrame(cv::Mat frame, cv::Mat &odd_frame, cv::Mat
 * &even_frame)
 * @brief Function to deinterlace the current image.
 *
 * @param[in] frame image to be processed;
 * @param[out] odd_frame odd plane;
 * @param[out] even_frame even plane.
 */
void separateFrame(const cv::Mat frame, cv::Mat& odd_frame, cv::Mat& even_frame);

/**
 * @fn void separateFrame(cv::Mat frame, cv::Mat &odd_frame, cv::Mat
 * &even_frame)
 * @brief Compute the number of different pixels between two frames.
 *
 * @param prevFrame the first frame;
 * @param currentFrame the second frame.
 * @return cv::Mat A black and white frame, where black pixels represent a
 * difference, while white pixels represent an equality.
 */
cv::Mat difference(cv::Mat& prevFrame, cv::Mat& currentFrame);
}  // namespace utility

/**
 * @struct Threshold
 * @brief Struct containing the threshold values used to detect a shape.
 *
 */
struct Threshold {
    float percentual; /**< The minimum percentage of different pixels for
                         considering the current frame under the ROI as a
                         potential Irregularity */
    int angle;        /**< The angle votes threshold for the detection of the object */
    int scale;        /**< The scale votes threshold for the detection of the object */
    int pos;          /**< The position votes threshold for the detection of the object
                       */

    Threshold(){};
    /**
     * @brief Construct a new Threshold object
     * @throws std::invalid_argument if the percentual is not in the range [0, 1]
     */
    Threshold(float percentual, int angle, int scale, int pos);
};

/**
 * @enum Object
 * @brief Enum containing the possible objects to detect.
 *
 */
enum Object { TAPE, CAPSTAN };

/**
 * @struct SceneObject
 * @brief A scene object is an object that can be detected in a scene, such as a
 * tape or a capstan.
 *
 */
struct SceneObject {
    int minDist;         /**< The minimum distance between the centers of the detected
                            objects for the detection of the reading head */
    Threshold threshold; /**<  the threshold values used to detect the object */

    SceneObject(){};
    SceneObject(int minDist, Threshold threshold);
    ~SceneObject() = default;
    /**
     * @fn static SceneObject from_file(fs::path path, Object obj)
     * @brief Create a SceneObject from a given file.
     *
     * @param path The path of the file containing the object.
     * @note The file must be a JSON file.
     * @param obj The object to detect.
     * @return SceneObject The SceneObject created from the file.
     */
    static SceneObject from_file(fs::path path, Object obj);
};
