#include <boost/lexical_cast.hpp>
#include <boost/uuid/uuid_io.hpp>
#include <boost/uuid/uuid.hpp>
#include <boost/uuid/uuid_generators.hpp>
#include "Irregularity.h"

Irregularity::Irregularity(const Irregularity& other)
    : id(other.id), source(other.source), time_label(other.time_label), type(other.type) {}

Irregularity::Irregularity(Irregularity&& other) noexcept
    : id(std::move(other.id)), source(other.source), time_label(std::move(other.time_label)), type(std::move(other.type)) {}

Irregularity::Irregularity(Source source, string time_label) {
    this->id = boost::uuids::random_generator()();
    this->source = source;
    this->time_label = time_label;
    this->type = std::nullopt;
}

Irregularity::Irregularity(Source source, string time_label, IrregularityType type) {
    this->id = boost::uuids::random_generator()();
    this->source = source;
    this->time_label = time_label;
    this->type = type;
}

json Irregularity::to_JSON() const {
    json j;

    j["IrregularityID"] = boost::lexical_cast<string>(this->id);
    j["Source"] = sourceToString(this->source);
    j["TimeLabel"] = this->time_label;
    if (this->type.has_value())
        j["IrregularityType"] = irregularityTypeToString(this->type.value());

    if (this->image_URI.has_value())
        j["ImageURI"] = this->image_URI.value();

    if (this->audio_URI.has_value())
        j["AudioURI"] = this->audio_URI.value();

    return j;
}

Irregularity Irregularity::from_JSON(const json& j) {
    Source source = sourceFromString(j["Source"]);
    string time_label = j["TimeLabel"];
    IrregularityType type = irregularityTypeFromString(j["IrregularityType"]);

    return Irregularity(source, time_label, type);
}

Source Irregularity::get_source() const {
    return this->source;
}
string Irregularity::get_time_label() const {
    return this->time_label;
}
std::optional<IrregularityType> Irregularity::get_type() const {
    return this->type;
}
boost::uuids::uuid Irregularity::get_id() const {
    return this->id;
}

std::optional<string> Irregularity::get_audio_URI() const {
    return this->audio_URI;
}

Irregularity& Irregularity::set_audio_URI(string audio_URI) {
    this->audio_URI = audio_URI;
    return *this;
}

std::optional<string> Irregularity::get_image_URI() const {
    return this->image_URI;
}

Irregularity& Irregularity::set_image_URI(string image_URI) {
    this->image_URI = image_URI;
    return *this;
}
