#ifndef UTILITY_H
#define UTILITY_H
#include <filesystem>
#include <opencv2/core/core.hpp>
#include <opencv2/highgui/highgui.hpp>
#include <opencv2/imgcodecs.hpp>
#include <opencv2/imgproc.hpp>

using namespace cv;
using namespace std;
namespace fs = std::filesystem;

/**
 * @brief Namespace containing a set of utility functions used in the project.
 * The functions are mainly used to perform operations on images.
 */
namespace utility {

/**
 * @fn void detect_shape(Ptr<GeneralizedHoughGuil> alg, int
 * pos_thresh, vector<Vec4f> &positive_positions, Mat &positive_votes,
 * vector<Vec4f> &negative_positions, Mat &negative_votes, Mat processing_area)
 * @brief Detects a shape in an image, using a the OpenCV algorithm
 * GeneralizedHoughGuil.
 *
 * @param[in] alg the algorithm instance;
 * @param[in] pos_thresh the position votes threshold, which determines the minimum number of votes required to consider
 * a detection valid;
 * @param[out] positive_positions vector representing the position assigned to
 * each found rectangle for positive angles;
 * @param[out] positive_votes vector representing the vote assigned to each found
 * rectangle for positive angles;
 * @param[out] negative_positions vector representing the position assigned to
 * each found rectangle for negative angles;
 * @param[out] negative_votes vector representing the vote assigned to each found
 * rectangle for negative angles;
 * @param[in] processing_area the image to be processed.
 */
void detect_shape(Ptr<GeneralizedHoughGuil> alg, int pos_thresh, vector<Vec4f>& positive_positions, Mat& positive_votes,
                  vector<Vec4f>& negative_positions, Mat& negative_votes, Mat processing_area);

/**
 * @fn RotatedRect drawShapes(Mat frame, Vec4f &positions, Scalar color, int
 * width, int height, int offsetX, int offsetY, float processingScale)
 * @brief Draw rectangles on an image.
 *
 * @param frame Frame on which the rectangles will be drawn;
 * @param positions The position of the rectangle;
 * @param color The color of the rectangle;
 * @param width The width of the rectangle;
 * @param height The height of the rectangle;
 * @param offsetX X offset on the position of the rectangle;
 * @param offsetY Y offset on the position of the rectangle;
 * @param processingScale Scaling factor, useful for downsizing.
 * @return RotatedRect Object representing the drawn rectangle.
 */
RotatedRect drawShapes(Mat frame, const Vec4f& positions, Scalar color, int width, int height, int offsetX, int offsetY,
                       float processingScale);
}  // namespace utility

/**
 * @struct Threshold
 * @brief Struct containing the threshold values used to detect a shape.
 *
 */
struct Threshold {
    float percentual; /**< The minimum percentage of different pixels for
                         considering the current frame under the ROI as a
                         potential Irregularity */
    int angle;        /**< The angle votes threshold for the detection of the object */
    int scale;        /**< The scale votes threshold for the detection of the object */
    int pos;          /**< The position votes threshold for the detection of the object
                       */

    Threshold(){};
    /**
     * @brief Construct a new Threshold object
     * @throws std::invalid_argument if the percentual is not in the range [0, 1]
     */
    Threshold(float percentual, int angle, int scale, int pos);
};

/**
 * @enum ROI
 * @brief Enum containing the possible objects to detect.
 *
 */
enum class ROI { TAPE, CAPSTAN };

/**
 * @struct SceneObject
 * @brief A scene object is an object that can be detected in a scene, such as a
 * tape or a capstan.
 *
 */
struct SceneObject {
    int minDist;         /**< The minimum distance between the centers of the detected
                            objects for the detection of the reading head */
    Threshold threshold; /**<  the threshold values used to detect the object */

    SceneObject(){};
    SceneObject(int minDist, Threshold threshold);
    ~SceneObject() = default;
    /**
     * @fn static SceneObject from_file(fs::path path, Object obj)
     * @brief Create a SceneObject from a given file.
     *
     * @param path The path of the file containing the object.
     * @note The file must be a JSON file.
     * @param obj The object to detect.
     * @return SceneObject The SceneObject created from the file.
     */
    static SceneObject from_file(fs::path path, ROI obj);
};
#endif  // UTILITY_H