#include "core.hpp"

namespace videoanalyser {
namespace core {

Frame::Frame() : cv::Mat() {}
Frame::Frame(const cv::Mat& m) : cv::Mat(m) {}
Frame::Frame(const Frame& f) : cv::Mat(f) {}
Frame& Frame::operator=(const Mat& m) {
    Mat::operator=(m);
    return *this;
}
Frame& Frame::operator=(const Frame& f) {
    Mat::operator=(f);
    return *this;
}

Frame Frame::clone() const { return Frame(cv::Mat::clone()); }
Frame& Frame::downsample(int factor) {
    cv::pyrDown(*this, *this, cv::Size(size().width / factor, size().height / factor));
    return *this;
}
Frame& Frame::convert_color(int code) {
    cv::cvtColor(*this, *this, code);
    return *this;
}
Frame Frame::difference(Frame& f) {
    Frame diff = this->clone();
    for (int i = 0; i < this->rows; i++) {
        for (int j = 0; j < this->cols; j++) {
            if (f.at<cv::Vec3b>(i, j)[0] != this->at<cv::Vec3b>(i, j)[0] ||
                f.at<cv::Vec3b>(i, j)[1] != this->at<cv::Vec3b>(i, j)[1] ||
                f.at<cv::Vec3b>(i, j)[2] != this->at<cv::Vec3b>(i, j)[2]) {
                // Different pixels
                diff.at<cv::Vec3b>(i, j)[0] = 0;
            } else {
                // Identical pixels
                diff.at<cv::Vec3b>(i, j)[0] = 255;
            }
        }
    }
    return diff;
}
Frame& Frame::crop(cv::Size rect_size, cv::Point2f center) {
    cv::getRectSubPix(*this, rect_size, center, *this);
    return *this;
}
Frame& Frame::warp(cv::Mat rotationMatrix) {
    cv::warpAffine(*this, *this, rotationMatrix, this->size(), cv::INTER_CUBIC);
    return *this;
}
std::pair<Frame, Frame> Frame::deinterlace() const {
    Frame odd_frame(cv::Mat(this->rows / 2, this->cols, CV_8UC3));
    Frame even_frame(cv::Mat(this->rows / 2, this->cols, CV_8UC3));

    int i_odd_frame = 0;
    int i_even_frame = 0;

    for (int i = 0; i < this->rows; i++) {
        for (int j = 0; j < this->cols; j++) {
            if (i % 2 == 0) {
                even_frame.at<cv::Vec3b>(i_even_frame, j)[0] = this->at<cv::Vec3b>(i, j)[0];
                even_frame.at<cv::Vec3b>(i_even_frame, j)[1] = this->at<cv::Vec3b>(i, j)[1];
                even_frame.at<cv::Vec3b>(i_even_frame, j)[2] = this->at<cv::Vec3b>(i, j)[2];
            } else {
                odd_frame.at<cv::Vec3b>(i_odd_frame, j)[0] = this->at<cv::Vec3b>(i, j)[0];
                odd_frame.at<cv::Vec3b>(i_odd_frame, j)[1] = this->at<cv::Vec3b>(i, j)[1];
                odd_frame.at<cv::Vec3b>(i_odd_frame, j)[2] = this->at<cv::Vec3b>(i, j)[2];
            }
        }

        if (i % 2 == 0) {
            i_even_frame++;
        } else {
            i_odd_frame++;
        }
    }
    return std::make_pair(odd_frame, even_frame);
}
}  // namespace core
}  // namespace videoanalyser