#include <opencv2/core/core.hpp>
#include <opencv2/highgui/highgui.hpp>
#include <opencv2/imgcodecs.hpp>
#include <opencv2/imgproc.hpp>

using namespace cv;
using namespace std;
namespace fs = std::filesystem;

namespace utility {

	class Frame : Mat {
	public:
		Frame() : Mat() {}
		Frame(const Mat& m) : Mat(m) {}
		Frame(const Frame& f) : Mat(f) {}
		Frame& operator=(const Mat& m) {
			Mat::operator=(m);
			return *this;
		}
		Frame& operator=(const Frame& f) {
			Mat::operator=(f);
			return *this;
		}
		Frame clone() const {
			return Frame(Mat::clone());
		}
		Frame& downsample(int factor) {
			pyrDown(*this, *this, Size(size().width / factor, size().height / factor));
			return *this;
		}
		Frame& convertColor(int code) {
			cv::cvtColor(*this, *this, code);
			return *this;
		}
	};

	/**
	 * @brief Detects a given shape in an image, using a the OpenCV algorithm GeneralizedHoughGuil.
	 *
	 * @param[in] alg the algorithm instance;
	 * @param[in] templateShape the shape to detect;
	 * @param[in] posThresh the position votes threshold;
	 * @param[out] positivePositions vector representing the position assigned to each found rectangle for positive angles;
	 * @param[out] positiveVotes vector representing the vote assigned to each found rectangle for positive angles;
	 * @param[out] negativePositions vector representing the position assigned to each found rectangle for negative angles;
	 * @param[out] negativeVotes vector representing the vote assigned to each found rectangle for negative angles;
	 * @param[in] processingArea the image to be processed.
	 */
	void detectShape(Ptr<GeneralizedHoughGuil> alg, Mat templateShape, int posThresh, vector<Vec4f> &positivePositions, Mat &positiveVotes, vector<Vec4f> &negativePositions, Mat &negativeVotes, Mat processingArea);

	/**
	 * @brief Draw rectangles on an image.
	 *
	 * @param frame Frame on which the rectangles will be drawn;
	 * @param positions The position of the rectangle;
	 * @param color The color of the rectangle;
	 * @param width The width of the rectangle;
	 * @param height The height of the rectangle;
	 * @param offsetX X offset on the position of the rectangle;
	 * @param offsetY Y offset on the position of the rectangle;
	 * @param processingScale Scaling factor, useful for downsizing.
	 * @return RotatedRect Object representing the drawn rectangle.
	 */
	RotatedRect drawShapes(Mat frame, Vec4f &positions, Scalar color, int width, int height, int offsetX, int offsetY, float processingScale);


	/**
	 * @brief Function to deinterlace the current image.
	 *
	 * @param[in] frame image to be processed;
	 * @param[out] odd_frame odd plane;
	 * @param[out] even_frame even plane.
	 */
	void separateFrame(cv::Mat frame, cv::Mat &odd_frame, cv::Mat &even_frame);


	/**
	 * @brief Compute the number of different pixels between two frames.
	 *
	 * @param prevFrame the first frame;
	 * @param currentFrame the second frame.
	 * @return cv::Mat A black and white frame, where black pixels represent a difference, while white pixels represent an equality.
	 */
	cv::Mat difference(cv::Mat &prevFrame, cv::Mat &currentFrame);

}
