from fastapi import FastAPI, Response, status, Path, Query
from typing import Annotated

from mpai_cae_arp.files import get_file_content
from mpai_cae_arp.types.irregularity import IrregularityFile
from mpai_cae_arp.types.schema import Info
import server.utils as utils

info = get_file_content('server/documentation.yaml', 'yaml')
app = FastAPI(**info)


@app.get("/")
async def list_endpoints() -> list[str]:
    return ["irregularityFile/{id}", "description", "docs"]


@app.get("/description")
async def get_server_info() -> Info:
    return info


@app.get("/irregularityFile/{id}", status_code=200)
async def get_irregularity_file(
        response: Response,
        id: Annotated[int, Path(
            gt=0, lt=3,
            description="Id of the irregularity file to be returned."
        )],
        files_name: Annotated[str, Query(
            description="Name of the video file to be analyzed without the extension.",
            example="video"
        )],
        working_path: Annotated[str, Query(
            description="Path to the directory containing the preservation files.",
            examples={
                "Absolute path": {"value": "/home/user/preservation"},
                "Relative path": {"value": "preservation"}
            }
        )] = None
    ) -> IrregularityFile:

    required_file = f'VideoAnalyser_IrregularityFileOutput{id}.json'
    
    try:
        return get_file_content(required_file, 'json')

    except:
        process = utils.analyze(files_name, working_path)

        if process.returncode == 0:
            return get_file_content(required_file, 'json')

        else:
            response.status_code = status.HTTP_412_PRECONDITION_FAILED
            return {"error": {
                "returncode": process.returncode,
                "stdout": process.stdout.decode("utf-8"),
                "stderr": process.stderr.decode("utf-8"),
            }}
