/**
 * @file Irregularity.h
 * @author Matteo Spanio (dev2@audioinnova.com)
 * @brief Header file containing the Irregularity class
 * @version 1.0
 * @date 2023-05-14
 *
 * @copyright Copyright (c) 2023
 *
 */
#ifndef IRREGULARITY_H
#define IRREGULARITY_H
#include <boost/uuid/uuid.hpp>
#include <nlohmann/json.hpp>

#include "enums.h"

using std::string;
using json = nlohmann::json;

/**
 * @class Irregularity
 * @brief an irregularity of the tape detected by the system
 *
 */
class Irregularity {
   private:
    boost::uuids::uuid id;
    Source source;
    string time_label;
    std::optional<IrregularityType> type;
    std::optional<string> image_URI;
    std::optional<string> audio_URI;

   public:
    Irregularity(const Irregularity& other);
    Irregularity(Irregularity&& other) noexcept;
    Irregularity(Source source, string time_label);
    Irregularity(Source source, string time_label, IrregularityType type);
    ~Irregularity() = default;
    /**
     * @brief Convert the Irregularity to a JSON object
     *
     * @return json
     */
    json to_JSON() const;
    /**
     * @brief Create an Irregularity object from a JSON object
     *
     * @param j the JSON object
     * @return Irregularity
     */
    static Irregularity from_JSON(const json& j);
    /**
     * @brief Get the source object
     *
     * @return Source
     */
    Source get_source() const;
    /**
     * @brief Get the time label object
     *
     * @return string
     */
    string get_time_label() const;
    /**
     * @brief Get the type object
     *
     * @return IrregularityType
     */
    std::optional<IrregularityType> get_type() const;
    /**
     * @brief Get the id object
     *
     * @return boost::uuids::uuid
     */
    boost::uuids::uuid get_id() const;
    /**
     * @brief Get the audio URI object
     *
     * @return std::optional<string>
     */
    std::optional<string> get_audio_URI() const;
    /**
     * @brief Get the image URI object
     *
     * @return std::optional<string>
     */
    std::optional<string> get_image_URI() const;
    /**
     * @brief Set the audio URI object
     *
     * @param audio_URI
     * @return Irregularity&
     */
    Irregularity& set_audio_URI(string audio_URI);
    /**
     * @brief Set the image URI object
     *
     * @param image_URI
     * @return Irregularity&
     */
    Irregularity& set_image_URI(string image_URI);
};

#endif  // IRREGULARITY_H