#include <opencv2/core/core.hpp>
#include <opencv2/highgui/highgui.hpp>
#include <opencv2/imgcodecs.hpp>
#include <opencv2/imgproc.hpp>

using namespace cv;
using namespace std;
namespace fs = std::filesystem;

/**
 * @brief Namespace containing a set of utility functions used in the project.
 * The functions are mainly used to perform operations on images.
 * 
 */
namespace utility {

	/**
	 * @class Frame
	 * @brief Class that extends the OpenCV Mat class, adding some useful methods frequently used in the project.
	 * 
	 */
	class Frame : public Mat {
	public:
		Frame();
		Frame(const Mat& m);
		Frame(const Frame& f);
		Frame& operator=(const Mat& m);
		Frame& operator=(const Frame& f);
		Frame clone() const;
		Frame& downsample(int factor);
		Frame& convertColor(int code);
		Frame difference(Frame& f);
		Frame& crop(Size rect_size, Point2f center);
		Frame& warp(cv::Mat rotationMatrix);
	};


	/**
	 * @fn void detectShape(Ptr<GeneralizedHoughGuil> alg, Mat templateShape, int posThresh, vector<Vec4f> &positivePositions, Mat &positiveVotes, vector<Vec4f> &negativePositions, Mat &negativeVotes, Mat processingArea)
	 * @brief Detects a given shape in an image, using a the OpenCV algorithm GeneralizedHoughGuil.
	 *
	 * @param[in] alg the algorithm instance;
	 * @param[in] templateShape the shape to detect;
	 * @param[in] posThresh the position votes threshold;
	 * @param[out] positivePositions vector representing the position assigned to each found rectangle for positive angles;
	 * @param[out] positiveVotes vector representing the vote assigned to each found rectangle for positive angles;
	 * @param[out] negativePositions vector representing the position assigned to each found rectangle for negative angles;
	 * @param[out] negativeVotes vector representing the vote assigned to each found rectangle for negative angles;
	 * @param[in] processingArea the image to be processed.
	 */
	void detectShape(Ptr<GeneralizedHoughGuil> alg, Mat templateShape, int posThresh, vector<Vec4f> &positivePositions, Mat &positiveVotes, vector<Vec4f> &negativePositions, Mat &negativeVotes, Mat processingArea);

	/**
	 * @fn RotatedRect drawShapes(Mat frame, Vec4f &positions, Scalar color, int width, int height, int offsetX, int offsetY, float processingScale)
	 * @brief Draw rectangles on an image.
	 *
	 * @param frame Frame on which the rectangles will be drawn;
	 * @param positions The position of the rectangle;
	 * @param color The color of the rectangle;
	 * @param width The width of the rectangle;
	 * @param height The height of the rectangle;
	 * @param offsetX X offset on the position of the rectangle;
	 * @param offsetY Y offset on the position of the rectangle;
	 * @param processingScale Scaling factor, useful for downsizing.
	 * @return RotatedRect Object representing the drawn rectangle.
	 */
	RotatedRect drawShapes(Mat frame, Vec4f &positions, Scalar color, int width, int height, int offsetX, int offsetY, float processingScale);


	/**
	 * @fn void separateFrame(cv::Mat frame, cv::Mat &odd_frame, cv::Mat &even_frame)
	 * @brief Function to deinterlace the current image.
	 *
	 * @param[in] frame image to be processed;
	 * @param[out] odd_frame odd plane;
	 * @param[out] even_frame even plane.
	 */
	void separateFrame(cv::Mat frame, cv::Mat &odd_frame, cv::Mat &even_frame);


	/**
	 * @fn void separateFrame(cv::Mat frame, cv::Mat &odd_frame, cv::Mat &even_frame)
	 * @brief Compute the number of different pixels between two frames.
	 *
	 * @param prevFrame the first frame;
	 * @param currentFrame the second frame.
	 * @return cv::Mat A black and white frame, where black pixels represent a difference, while white pixels represent an equality.
	 */
	cv::Mat difference(cv::Mat &prevFrame, cv::Mat &currentFrame);
}
