/**
 * @file enums.h
 * @author Matteo Spanio (dev2@audioinnova.com)
 * @brief A collection of enums and functions to handle them.
 * @version 1.0
 * @date 2023-05-13
 * 
 * @copyright Copyright (c) 2023
 * 
 * This file contains a set of enums to define properties of an Irregularity.
 * The enums are:
 * - Source: the source of the Irregularity (Audio, Video or Both)
 * - IrregularityType: the type of Irregularity (Brands on tape, Splice, etc.) that can be present on the tape.
 * 
 * The file also contains functions to convert from enum to string and viceversa (useful when saving the Irregularity to a file).
 * 
 */
#ifndef ENUMS_H
#define ENUMS_H
#include<string>

/**
 * @enum Source
 * @brief The source of the Irregularity (Audio, Video or Both)
 * 
 * An Irregularity can be detected by the Audio analyser, the Video analyser or both.
 * 
 */
enum Source{
    Audio,
    Video,
    Both
};

/**
 * @enum IrregularityType
 * @brief The type of Irregularity (Brands on tape, Splice, etc.) that can be present on the tape.
 * 
 * The types of Irregularities are:
 * - BRANDS_ON_TAPE: Brands on tape
 * - SPLICE: Splice
 * - START_OF_TAPE: Start of tape
 * - ENDS_OF_TAPE: End of tape
 * - DAMAGED_TAPE: Damaged tape
 * - DIRT: Dirt
 * - MARKS: Marks
 * - SHADOWS: Shadows
 * - WOW_AND_FLUTTER: Wow and flutter
 * - PLAY_PAUSE_STOP: Play, pause, stop
 * - SPEED: Speed
 * - EQUALIZATION: Equalization
 * - SPEED_AND_EQUALIZATION: Speed and equalization
 * - BACKWARD: Backward
 * 
 * @note Speed, Equalization and Speed and equalization are detected only by the Audio analyser,
 * while the other Irregularities are detected only by the Video analyser.
 * 
 */
enum IrregularityType {
    BRANDS_ON_TAPE,
    SPLICE,
    START_OF_TAPE,
    ENDS_OF_TAPE,
    DAMAGED_TAPE,
    DIRT,
    MARKS,
    SHADOWS,
    WOW_AND_FLUTTER,
    PLAY_PAUSE_STOP,
    SPEED,
    EQUALIZATION,
    SPEED_AND_EQUALIZATION,
    BACKWARD
};

std::string sourceToString(Source source);
Source sourceFromString(std::string source);
std::string irregularityTypeToString(IrregularityType type);
IrregularityType irregularityTypeFromString(std::string type);
#endif // ENUMS_H