/**
 * @file IrregularityFile.h
 * @author Matteo Spanio (dev2@audioinnova.com)
 * @brief Header file containing the IrregularityFile class
 * @version 1.0
 * @date 2023-05-14
 * 
 * @copyright Copyright (c) 2023
 * 
 */
#ifndef IRREGULARITY_FILE_H
#define IRREGULARITY_FILE_H

#include <vector>
#include <nlohmann/json.hpp>
#include "Irregularity.h"
#include <optional>

using json = nlohmann::json;

/**
 * @class IrregularityFile
 * @brief An IrregularityFile is a collection of Irregularities detected on a tape.
 * 
 */
class IrregularityFile {
public:
    /**
     * @brief Create an IrregularityFile object from a JSON object
     * 
     * @param j 
     * @return IrregularityFile 
     */
    static IrregularityFile fromJSON(const json j);
    /**
     * @brief Convert the IrregularityFile to a JSON object
     * 
     * @return json 
     */
    json toJSON() const;
    IrregularityFile(std::optional<uint16_t> offset = std::nullopt);
    /**
     * @brief Copy constructor
     * 
     * @param rhs 
     */
    IrregularityFile(const IrregularityFile &rhs);
    ~IrregularityFile() {};
    /**
     * @brief Add an Irregularity to the IrregularityFile
     * 
     * @param irregularity 
     * @return IrregularityFile& 
     */
    IrregularityFile& add(std::unique_ptr<Irregularity> irregularity);
    /**
     * @brief Remove an Irregularity from the IrregularityFile
     * 
     * @param id 
     * @return IrregularityFile& 
     */
    IrregularityFile& remove_by_id(const boost::uuids::uuid id);
    /**
     * @brief Sort the IrregularityFile by time_label
     * 
     * @return IrregularityFile& 
     */
    IrregularityFile& sort();
    /**
     * @brief Get the offset object
     * 
     * @return std::optional<uint16_t> 
     */
    std::optional<uint16_t> get_offset() const;
    /**
     * @brief Get an iterator to the beginning of the IrregularityFile
     * 
     * @return std::vector<std::unique_ptr<Irregularity>>::iterator 
     */
    std::vector<std::unique_ptr<Irregularity>>::iterator begin();
    /**
     * @brief Get an iterator to the end of the IrregularityFile
     * 
     * @return std::vector<std::unique_ptr<Irregularity>>::iterator 
     */
    std::vector<std::unique_ptr<Irregularity>>::iterator end();

private:
    std::optional<uint16_t> offset_;
    std::vector<std::unique_ptr<Irregularity>> irregularities_;
};

#endif // IRREGULARITY_FILE_HPP
