#ifndef UTILITY_H
#define UTILITY_H
#include <filesystem>
#include <fstream>
#include <nlohmann/json.hpp>

using namespace std;
namespace fs = std::filesystem;

/**
 * @struct Threshold
 * @brief Struct containing the threshold values used to detect a shape.
 *
 */
struct Threshold {
    float percentual; /**< The minimum percentage of different pixels for
                         considering the current frame under the ROI as a
                         potential Irregularity */
    int angle;        /**< The angle votes threshold for the detection of the object */
    int scale;        /**< The scale votes threshold for the detection of the object */
    int pos;          /**< The position votes threshold for the detection of the object
                       */

    Threshold(){};
    /**
     * @brief Construct a new Threshold object
     * @throws std::invalid_argument if the percentual is not in the range [0, 1]
     */
    Threshold(float percentual, int angle, int scale, int pos);
};

/**
 * @enum ROI
 * @brief Enum containing the possible objects to detect.
 *
 */
enum class ROI { TAPE, CAPSTAN };

/**
 * @struct SceneObject
 * @brief A scene object is an object that can be detected in a scene, such as a
 * tape or a capstan.
 *
 */
struct SceneObject {
    int minDist;         /**< The minimum distance between the centers of the detected
                            objects for the detection of the reading head */
    Threshold threshold; /**<  the threshold values used to detect the object */

    SceneObject(){};
    SceneObject(int minDist, Threshold threshold);
    ~SceneObject() = default;
    /**
     * @fn static SceneObject from_file(fs::path path, Object obj)
     * @brief Create a SceneObject from a given file.
     *
     * @param path The path of the file containing the object.
     * @note The file must be a JSON file.
     * @param obj The object to detect.
     * @return SceneObject The SceneObject created from the file.
     */
    static SceneObject from_file(fs::path path, ROI obj);
};
#endif  // UTILITY_H