#include "IrregularityFile.hpp"

IrregularityFile::IrregularityFile(std::optional<uint16_t> offset) : offset_(offset) {}

IrregularityFile::IrregularityFile(const IrregularityFile& rhs) {
    std::transform(rhs.irregularities_.begin(), rhs.irregularities_.end(), std::back_inserter(irregularities_),
                   [](const std::unique_ptr<Irregularity>& ptr) { return std::make_unique<Irregularity>(*ptr); });
}

IrregularityFile& IrregularityFile::add(std::unique_ptr<Irregularity> irregularity) {
    irregularities_.push_back(std::move(irregularity));
    return *this;
}

IrregularityFile& IrregularityFile::remove_by_id(const boost::uuids::uuid id) {
    auto it =
        std::find_if(irregularities_.begin(), irregularities_.end(),
                     [&id](const std::unique_ptr<Irregularity>& irregularity) { return irregularity->get_id() == id; });
    if (it != irregularities_.end()) {
        irregularities_.erase(it);
    }
    return *this;
}

IrregularityFile& IrregularityFile::sort() {
    std::sort(irregularities_.begin(), irregularities_.end(),
              [](const std::unique_ptr<Irregularity>& a, const std::unique_ptr<Irregularity>& b) {
                  return a->get_time_label() < b->get_time_label();
              });
    return *this;
}

std::optional<uint16_t> IrregularityFile::get_offset() const { return offset_; }

std::vector<std::unique_ptr<Irregularity>>::iterator IrregularityFile::begin() { return irregularities_.begin(); }

std::vector<std::unique_ptr<Irregularity>>::iterator IrregularityFile::end() { return irregularities_.end(); }

json IrregularityFile::toJSON() const {
    json j;
    j["Offset"] = offset_.value_or(0);
    j["Irregularities"] = json::array();
    for (const auto& irregularity : irregularities_) {
        j["Irregularities"].push_back(irregularity->to_JSON());
    }
    return j;
}

IrregularityFile IrregularityFile::fromJSON(const json j) {
    if (!j.contains("Offset") || !j.contains("Irregularities")) {
        throw std::invalid_argument("Invalid JSON");
    }
    IrregularityFile irregularity_file(j["Offset"].get<uint16_t>());
    for (const auto& irregularity : j["Irregularities"]) {
        irregularity_file.add(std::make_unique<Irregularity>(Irregularity::from_JSON(irregularity)));
    }
    return irregularity_file;
}
