import os
from concurrent import futures
import grpc
from grpc import StatusCode
from rich.console import Console
from scipy.io import wavfile

from mpai_cae_arp.audio.standards import EqualizationStandard, SpeedStandard
from mpai_cae_arp.files import File, FileType
from mpai_cae_arp.time import seconds_to_string
from mpai_cae_arp.types.restoration import Restoration, EditingList
from mpai_cae_arp.network import arp_pb2_grpc as arp_pb2_grpc
from mpai_cae_arp.network.arp_pb2 import (
    JobRequest,
    JobResponse,
    Contact,
    InfoResponse,
    License,
)

from tape_audio_restoration import lib

PORT = os.getenv("PORT") or '50051'
info = File('config.yml', FileType.YAML).get_content()


def error_response(context, status, message):
    context.set_code(status)
    context.set_details(message)
    return JobResponse(status="error", message=message)


class TapeAudioRestorationServicer(arp_pb2_grpc.AIMServicer):

    def __init__(self, console: Console):
        self.console = console

    def getInfo(self, request, context) -> InfoResponse:
        self.console.log('Received request for AIM info')

        context.set_code(StatusCode.OK)
        context.set_details('Success')

        return InfoResponse(
            title=info['title'],
            description=info['description'],
            version=info['version'],
            contact=Contact(
                name=info['contact']['name'],
                email=info['contact']['email'],
            ),
            license=License(
                name=info['license_info']['name'],
                url=info['license_info']['url'],
            )
        )

    def work(self, request: JobRequest, context):
        self.console.log('Received request for computation')
        self.console.log(request)
        
        working_dir: str = request.working_dir
        files_name: str = request.files_name

        temp_dir = os.path.join(working_dir, "temp", files_name)

        audio_irreg_2 = os.path.join(temp_dir,'AudioAnalyser_IrregularityFileOutput2.json' )
        editing_list_path = os.path.join(temp_dir,'EditingList.json' )

        try:
            standard_w, speed_w, standard_r, speed_r = lib.read_params_from_irregularity_file(audio_irreg_2)
            paf_path, temp_path, standard_w, standard_r = lib.check_input(working_dir, files_name, standard_w, speed_w, standard_r, speed_r)
            yield JobResponse(status="info", message=f"Applying equalisation filter from {standard_w} to {standard_r}")
            yield JobResponse(status="info", message=f"Applying speed filter from {speed_w} to {speed_r}")

            sample_rate, preservation_audio_file = wavfile.read(paf_path)

            correction_filter = lib.get_correction_filter(standard_w, speed_w, standard_r, speed_r, sample_rate)

            restored_audio_file = preservation_audio_file
            if correction_filter is None:
                yield JobResponse(status="success", message="No correction filter has to be applied")
                new_sample_rate = sample_rate
            else:
                a, b, new_sample_rate, _ = correction_filter
                yield JobResponse(status="info", message=f"Converting from {sample_rate}Hz to {new_sample_rate}Hz")
                
                if len(a) != 0:
                    restored_audio_file = lib.correction(a, b, preservation_audio_file, round(new_sample_rate))
            lib.save_file(restored_audio_file, round(new_sample_rate), temp_path, files_name)

            yield JobResponse(status="success", message=f"Restored audio file created at {temp_path}/RestoredAudioFile/{files_name}.wav")

            editing_list = EditingList(
                OriginalEqualisationStandard=standard_w,
                OriginalSampleFrequency=sample_rate,
                OriginalSpeedStandard=SpeedStandard(speed_w),
                Restorations=[
                    Restoration(
                        PreservationAudioFileStart="00:00:00.000",
                        PreservationAudioFileEnd=seconds_to_string(len(preservation_audio_file) / sample_rate),
                        RestoredAudioFileURI=f"{temp_path}/RestoredAudioFile/{files_name}.wav",
                        ReadingBackwards=False,
                        AppliedSpeedStandard=SpeedStandard(speed_r),
                        AppliedSampleFrequency=new_sample_rate,
                        AppliedEqualisationStandard=standard_r,
                    )
                ]
            )

            editing_list.save_as_json_file(editing_list_path)
            yield JobResponse(status="success", message=f"Editing list created at {editing_list_path}")

            yield JobResponse(status="success", message="Success")
        except Exception as e:
            yield JobResponse(status="error", message=str(e))


def serve(console):
    server = grpc.server(futures.ThreadPoolExecutor(max_workers=10))
    arp_pb2_grpc.add_AIMServicer_to_server(TapeAudioRestorationServicer(console), server)
    server.add_insecure_port(f'[::]:{PORT}')
    server.start()
    server.wait_for_termination()


if __name__ == '__main__':
    console = Console()
    console.print(f'Server started at localhost:{PORT} :satellite:')
    serve(console)
