"""
MPAI CAE-ARP Tape Audio Restoration.

Implements MPAI CAE-ARP Tape Audio Restoration Technical Specification.
It identifies and restore portions of the Preservation Audio File, providing:
- Restored Audio Files;
- Editing List
"""

import array
import numpy as np
import os
import shutil
from statistics import mode
from control import c2d, TransferFunction
from numpy import ndarray
from scipy.io import wavfile
from scipy.signal import tf2zpk, zpk2tf, lfilter

from mpai_cae_arp.audio.standards import EqualizationStandard, SpeedStandard
from mpai_cae_arp.files import File, FileType
from mpai_cae_arp.io import Color, pprint
from mpai_cae_arp.types.irregularity import IrregularityFile


__copyright__ = "Copyright 2022, Audio Innova S.r.l."
__credits__ = ["Niccolò Pretto", "Nadir Dalla Pozza", "Sergio Canazza"]
__status__ = "Production"


def verify_path(working_path: str, files_name: str) -> str:
    raise NotImplementedError


def verify_equalization_and_speed(standard_w: EqualizationStandard | str, speed_w: SpeedStandard | float, standard_r: EqualizationStandard | str, speed_r: SpeedStandard | float) -> tuple[str, str]:
    """
    Method to verify that the equalization and speed parameters are correct.
    """
    raise NotImplementedError


def check_input(working_path: str, files_name: str, standard_w: EqualizationStandard, speed_w: float, standard_r: EqualizationStandard, speed_r: float) -> tuple[str, str, EqualizationStandard, EqualizationStandard]:
    """
    Method to check that passed arguments are correct and that the environment is conformant to the standard;

    Parameters
    ----------
    working_path: str
        representing the path where all files resulting from previous AIMs are stored,
    files_name: str
        representing the Preservation files name, to identify the input directory,
    standard_w: EqualizationStandard
        specifying the equalization standard used when the tape was recorded,
    speed_w: float
        specifying the speed used when the tape was recorded,
    standard_r: EqualizationStandard
        specifying the equalization standard used when the tape was read,
    speed_r: float
        specifying the speed used when the tape was read.
        
    Returns
    -------
    tuple[str, str, EqualizationStandard, EqualizationStandard]
        consisting of four variables:
        - str representing the path where the Preservation Audio File is stored;
        - str representing the path where the files to be processed during the current execution are stored;
        - the operating standard_w;
        - the operating standard_r.
    """
    if standard_w is EqualizationStandard.CCIR:
        standard_w = 'CCIR'
    elif standard_w is EqualizationStandard.NAB:
        standard_w = 'NAB'
    elif standard_w is EqualizationStandard.IEC:
        standard_w = 'IEC'
    
    if standard_r is EqualizationStandard.CCIR:
        standard_r = 'CCIR'
    elif standard_r is EqualizationStandard.NAB:
        standard_r = 'NAB'
    elif standard_r is EqualizationStandard.IEC:
        standard_r = 'IEC'

    # Check for working path existence
    if not os.path.exists(working_path):
        pprint('The specified WORKING_PATH is non-existent!', color=Color.RED)
        quit(os.EX_CONFIG)
    # Check for Preservation Audio File existence
    audio_file = files_name + '.wav'
    paf_path = os.path.join(working_path, 'PreservationAudioFile', audio_file)
    if not os.path.exists(paf_path):
        pprint('Preservation Audio File not found!', color=Color.RED)
        quit(os.EX_NOINPUT)
    # Check for temp directory existence
    temp_path = os.path.join(working_path, 'temp')
    if not os.path.exists(temp_path):
        pprint('WORKING_PATH structure is not conformant!', color=Color.RED)
        quit(os.EX_NOINPUT)
    # Check for input directory existence
    temp_path = os.path.join(temp_path, files_name)
    if not os.path.exists(temp_path):
        pprint('WORKING_PATH structure is not conformant!', color=Color.RED)
        quit(os.EX_NOINPUT)

    # Configuration parameters check

    # Recording tape speed check
    if speed_w != 3.75 and speed_w != 7.5 and speed_w != 15 and speed_w != 30:
        pprint(
            f"Incorrect SPEED_W: '{speed_w}'. Accepted value are: 3.75, 7.5, 15, 30.",
            color=Color.RED
        )
        quit(os.EX_CONFIG)

    # Reading tape speed check.
    if speed_r != 3.75 and speed_r != 7.5 and speed_r != 15 and speed_r != 30:
        pprint(
            f"Incorrect SPEED_R: '{speed_r}'. Accepted value are: 3.75, 7.5, 15, 30.",
            color=Color.RED
        )
        quit(os.EX_CONFIG)

    # Equalization standard check.
    if not (standard_r == 'CCIR' or standard_r == 'NAB'):
        pprint(
            f"Incorrect STANDARD_R: '{standard_r}'. Accepted values are: CCIR, NAB.",
            color=Color.RED
        )
        quit(os.EX_CONFIG)
    if not (standard_w == 'CCIR' or standard_w == 'NAB'):
        pprint(
            f"Incorrect STANDARD_W: '{standard_w}'. Accepted values are: CCIR, NAB.",
            color=Color.RED
        )
        quit(os.EX_CONFIG)

    # CCIR speed check.
    if (standard_w == 'CCIR' and speed_w == 3.75):
        pprint(
            'CCIR is undefined at 3.75 ips. Recording equalization standard is set to NAB.',
            color=Color.YELLOW
        )
        standard_w = 'NAB'
    if standard_r == 'CCIR' and speed_r == 3.75:
        pprint(
            'CCIR is undefined at 3.75 ips. Reading equalization standard is set to NAB.',
            color=Color.YELLOW
        )
        standard_r = 'NAB'
    # NAB speed check.
    if standard_w == 'NAB' and speed_w == 30:
        pprint(
            'NAB is undefined at 30 ips. Recording equalization standard is set to CCIR.',
            color=Color.YELLOW
        )
        standard_w = 'CCIR'
    if standard_r == 'NAB' and speed_r == 30:
        pprint(
            'NAB is undefined at 30 ips. Reading equalization standard is set to CCIR.',
            color=Color.YELLOW
        )
        standard_r = 'CCIR'

    if standard_w == 'CCIR':
        standard_w = EqualizationStandard.CCIR
    elif standard_w == 'NAB':
        standard_w = EqualizationStandard.NAB
        
    if standard_r == 'CCIR':
        standard_r = EqualizationStandard.CCIR
    elif standard_r == 'NAB':
        standard_r = EqualizationStandard.NAB

    return paf_path, temp_path, standard_w, standard_r


def get_correction_filter(standard_w: EqualizationStandard, speed_w: float, standard_r: EqualizationStandard, speed_r: float, fs: int) -> tuple[list, list, float, int] | None:
    """
    Method to establish correct filter transfer function coefficients;

    Parameters
    ----------
    standard_w : EqualizationStandard
        String specifying the equalization standard used when the tape was recorded.
    speed_w : float
        Float specifying the speed used when the tape was recorded.
    standard_r : EqualizationStandard
        String specifying the equalization standard used when the tape was read.
    speed_r : float
        Float specifying the speed used when the tape was read.
    fs : int
        Integer specifying the sampling frequency.
    
    Returns
    -------
    tuple
        Tuple consisting of four variables:

        * list of the filter numerator coefficients;
        * list of the filter denominator coefficients;
        * float specifying the operating sampling frequency;
        * int informing about the case number.
    """

    if standard_w is EqualizationStandard.CCIR:
        standard_w = 'CCIR'
    elif standard_w is EqualizationStandard.NAB:
        standard_w = 'NAB'
    elif standard_w is EqualizationStandard.IEC:
        standard_w = 'IEC'
    
    if standard_r is EqualizationStandard.CCIR:
        standard_r = 'CCIR'
    elif standard_r is EqualizationStandard.NAB:
        standard_r = 'NAB'
    elif standard_r is EqualizationStandard.IEC:
        standard_r = 'IEC'
    
    # CCIR time constants.
    t2_30 = 17.5 * 10 ** (-6)  # time constant CCIR_30
    t2_15 = 35 * 10 ** (-6)  # time constant CCIR_15
    t2_7 = 70 * 10 ** (-6)  # time constant CCIR_7.5

    # NAB time constants.
    t3 = 3180 * 10 ** (-6)
    t4_15 = 50 * 10 ** (-6)  # time constant NAB_15
    t4_7 = 50 * 10 ** (-6)  # time constant NAB_7.5
    t4_3 = 90 * 10 ** (-6)  # time constant NAB_3.75

    a = []
    b = []
    case = -1
    # This section will establish which time constants must be modified to obtain the desired equalisation standard.
    if standard_w == 'CCIR':
        if speed_w == 30:
            if standard_r == 'NAB':
                # Case 1
                if speed_r == 15:
                    fs = 2 * fs  # Doubling the sampling frequency
                    # Correction filter: NABw15_mod + CCIRr30
                    # - NAB constants divided by 2
                    t3 = t3 / 2
                    t4 = t4_15 / 2
                    # - CCIR_30 constant not altered
                    t2 = t2_30
                    # Filter coefficients
                    a = [t2 * t3, t2 + t3, 1]
                    b = [t3 * t4, t3, 0]
                    # Plot information
                    case = 1
                # Case 2
                elif speed_r == 7.5:
                    fs = 4 * fs  # Quadrupling the sampling frequency
                    # Correction filter: NABw7.5_mod + CCIRr30
                    # - NAB constants divided by 4
                    t3 = t3 / 4
                    t4 = t4_7 / 4
                    # - CCIR_30 constant not altered
                    t2 = t2_30
                    # Filter coefficients
                    a = [t2 * t3, t2 + t3, 1]
                    b = [t3 * t4, t3, 0]
                    # Plot information
                    case = 2
                # Case 3
                else:  # speed_r == 3.75
                    fs = 8 * fs  # Multiplying by 8 the sampling frequency
                    # Correction filter: NABw3.75_mod + CCIRr30
                    # - NAB constants divided by 8
                    t3 = t3 / 8
                    t4 = t4_3 / 8
                    # - CCIR_30 constant not altered
                    t2 = t2_30
                    # Filter coefficients
                    a = [t2 * t3, t2 + t3, 1]
                    b = [t3 * t4, t3, 0]
                    # Plot information
                    case = 3
            else:  # standard_r == 'CCIR'
                # Case 31
                if speed_r == 30:
                    print('Reference case: 31')
                    pprint('Nothing to do!', color=Color.GREEN)
                    return None
                # Case 15
                elif speed_r == 15:
                    fs = 2 * fs  # Doubling sampling frequency
                    # Plot information
                    case = 15
                # Case 16
                else:  # speed_r == 7.5
                    fs = 4 * fs  # Quadrupling the sampling frequency
                    # Plot information
                    case = 16
        elif speed_w == 15:
            if standard_r == 'NAB':
                # Case 28
                if speed_r == 15:
                    # No speed change
                    # Correction filter: NABw15 + CCIRr15
                    # - NAB_15 constants not altered
                    t4 = t4_15
                    # - CCIR_30 constant not altered
                    t2 = t2_15
                    # Filter coefficients
                    a = [t2 * t3, t2 + t3, 1]
                    b = [t3 * t4, t3, 0]
                    # Plot information
                    case = 28
                # Case 6
                elif speed_r == 7.5:
                    fs = 2 * fs  # Doubling the sampling frequency
                    # Correction filter: NABw7.5_mod + CCIRr15
                    # - NAB constants divided by 2
                    t3 = t3 / 2
                    t4 = t4_7 / 2
                    # - CCIR_15 constant not altered
                    t2 = t2_15
                    # Filter coefficients
                    a = [t2 * t3, t2 + t3, 1]
                    b = [t3 * t4, t3, 0]
                    # Plot information
                    case = 6
                # Case 7
                else:  # speed_r == 3.75
                    fs = 4 * fs  # Quadrupling the sampling frequency
                    # Correction filter: NABw3.75_mod + CCIRr15
                    # - NAB constants divided by 4
                    t3 = t3 / 4
                    t4 = t4_3 / 4
                    # - CCIR_15 constant not altered
                    t2 = t2_15
                    # Filter coefficients
                    a = [t2 * t3, t2 + t3, 1]
                    b = [t3 * t4, t3, 0]
                    # Plot information
                    case = 7
            else:  # standard_r == 'CCIR'
                # Case 19
                if speed_r == 30:
                    fs = fs / 2  # Halving the sampling frequency
                    # Plot information
                    case = 19
                # Case 33
                elif speed_r == 15:
                    print('Reference case: 33')
                    pprint('Nothing to do!', color=Color.GREEN)
                    return None
                # Case 20
                else:  # speed_r == 7.5
                    fs = 2 * fs  # Doubling the sampling frequency
                    # Plot information
                    case = 20
        else:  # speed_w == 7.5
            if standard_r == 'NAB':
                # Case 10
                if speed_r == 15:
                    fs = fs / 2  # Halving the sampling frequency
                    # Correction filter: NABw15_mod + CCIRr7.5
                    # - NAB constants multiplied by 2
                    t3 = t3 * 2
                    t4 = t4_15 * 2
                    # - CCIR_7.5 constant not altered
                    t2 = t2_7
                    # Filter coefficients
                    a = [t2 * t3, t2 + t3, 1]
                    b = [t3 * t4, t3, 0]
                    # Plot information
                    case = 10
                # Case 30
                elif speed_r == 7.5:
                    # No speed change
                    # Correction filter: NABw7.5 + CCIRr7.5
                    # - NAB_7.5 constant not altered
                    t4 = t4_7
                    # - CCIR_7.5 constant not altered
                    t2 = t2_7
                    # Filter coefficients
                    a = [t2 * t3, t2 + t3, 1]
                    b = [t3 * t4, t3, 0]
                    # Plot information
                    case = 30
                # Case 11
                else:  # speed_r == 3.75
                    fs = 2 * fs  # Doubling the sampling frequency
                    # Correction filter: NABw3.75_mod + CCIRr7.5
                    # - NAB constants divided by 2
                    t3 = t3 / 2
                    t4 = t4_3 / 2
                    # - CCIR_7.5 constant not altered
                    t2 = t2_7
                    # Filter coefficients
                    a = [t2 * t3, t2 + t3, 1]
                    b = [t3 * t4, t3, 0]
                    # Plot information
                    case = 11
            else:  # standard_r == 'CCIR'
                # Case 23
                if speed_r == 30:
                    fs = fs / 4  # Quartering the sampling frequency
                    # Plot information
                    case = 23
                # Case 24
                elif speed_r == 15:
                    fs = fs / 2  # Halving the sampling frequency
                    # Plot information
                    case = 24
                # Case 35
                else:  # speed_r == 7.5
                    print('Reference case: 35')
                    pprint('Nothing to do!', color=Color.GREEN)
                    return None
    else:  # standard_w == 'NAB'
        if speed_w == 15:
            if standard_r == 'NAB':
                # Case 32
                if speed_r == 15:
                    print('Reference case: 32')
                    pprint('Nothing to do!', color=Color.GREEN)
                    return None
                # Case 17
                elif speed_r == 7.5:
                    fs = 2 * fs  # Doubling the sampling frequency
                    # Correction filter: NABw7.5_mod + NABr15
                    # - NABw constants divided by 2
                    t3_mod = t3 / 2
                    t4_mod = t4_7 / 2
                    # - NABr constant not altered
                    t4 = t4_15
                    # Filter coefficients
                    a = [t3 * t3_mod * t4, t3 * (t3_mod + t4), t3]
                    b = [t3 * t3_mod * t4_mod, t3_mod * (t3 + t4_mod), t3_mod]
                    # Plot information
                    case = 17
                # Case 18
                else:  # speed_r == 3.75
                    fs = 4 * fs  # Quadrupling the sampling frequency
                    # Correction filter: NABw3.75_mod + NABr15
                    # - NAB constants divided by 4
                    t3_mod = t3 / 4
                    t4_mod = t4_3 / 4
                    # - NABr constant not altered
                    t4 = t4_15
                    # Filter coefficients
                    a = [t3 * t3_mod * t4, t3 * (t3_mod + t4), t3]
                    b = [t3 * t3_mod * t4_mod, t3_mod * (t3 + t4_mod), t3_mod]
                    # Plot information
                    case = 18
            else:  # standard_r == 'CCIR'
                # Case 4
                if speed_r == 30:
                    fs = fs / 2  # Halving the sampling frequency
                    # Correction filter: CCIRw30_mod + NABr15
                    # - CCIR_30 constant multiplied by 2
                    t2 = t2_30 * 2
                    # - NAB_15 constant not altered
                    t4 = t4_15
                    # Filter coefficients
                    a = [t3 * t4, t3, 0]
                    b = [t2 * t3, t2 + t3, 1]
                    # Plot information
                    case = 4
                # Case 27
                elif speed_r == 15:
                    # No speed change
                    # Correction filter: CCIRw15 + NABr15
                    # - CCIR_15 constant not altered
                    t2 = t2_15
                    # - NAB_15 constant not altered
                    t4 = t4_15
                    # Filter coefficients
                    a = [t3 * t4, t3, 0]
                    b = [t2 * t3, t2 + t3, 1]
                    # Plot information
                    case = 27
                # Case 5
                else:  # speed_r == 7.5
                    fs = fs * 2  # Doubling the sampling frequency
                    # Correction filter: CCIRw7.5_mod + NABr15
                    # - CCIR_7.5 constant divided by 2
                    t2 = t2_7 / 2
                    # - NAB_15 constant not altered
                    t4 = t4_15
                    # Filter coefficients
                    a = [t3 * t4, t3, 0]
                    b = [t2 * t3, t2 + t3, 1]
                    # Plot information
                    case = 5
        elif speed_w == 7.5:
            if standard_r == 'NAB':
                # Case 21
                if speed_r == 15:
                    fs = fs / 2  # Halving the sampling frequency
                    # Correction filter: NABw15_mod + NABr7.5
                    # - NABw constants multiplied by 2
                    t3_mod = t3 * 2
                    t4_mod = t4_15 * 2
                    # - NABr constant not altered
                    t4 = t4_7
                    # Filter coefficients
                    a = [t3 * t3_mod * t4, t3 * (t3_mod + t4), t3]
                    b = [t3 * t3_mod * t4_mod, t3_mod * (t3 + t4_mod), t3_mod]
                    # Plot information
                    case = 21
                # Case 34
                elif speed_r == 7.5:
                    print('Reference case: 34')
                    pprint('Nothing to do!', color=Color.GREEN)
                    return None
                # Case 22
                else:  # speed_r == 3.75
                    fs = 2 * fs  # Doubling the sampling frequency
                    # Correction filter: NABw3.75_mod + NABr7.5
                    # - NABw constants divided by 2
                    t3_mod = t3 / 2
                    t4_mod = t4_3 / 2
                    # - NABr constant not altered
                    t4 = t4_7
                    # Filter coefficients
                    a = [t3 * t3_mod * t4, t3 * (t3_mod + t4), t3]
                    b = [t3 * t3_mod * t4_mod, t3_mod * (t3 + t4_mod), t3_mod]
                    # Plot information
                    case = 22
            else:  # standard_r == 'CCIR'
                # Case 8
                if speed_r == 30:
                    fs = fs / 4  # Quartering the sampling frequency
                    # Correction filter: CCIRw30_mod + NABr7.5
                    # - CCIR_30 constant multiplied by 4
                    t2 = t2_30 * 4
                    # - NAB_7.5 constant not altered
                    t4 = t4_7
                    # Filter coefficients
                    a = [t3 * t4, t3, 0]
                    b = [t2 * t3, t2 + t3, 1]
                    # Plot information
                    case = 8
                # Case 9
                elif speed_r == 15:
                    fs = fs / 2  # Halving the sampling frequency
                    # Correction filter: CCIRw15_mod + NABr7.5
                    # - CCIR_15 constant multiplied by 2
                    t2 = t2_15 * 2
                    # - NAB_7.5 constant not altered
                    t4 = t4_7
                    # Filter coefficients
                    a = [t3 * t4, t3, 0]
                    b = [t2 * t3, t2 + t3, 1]
                    # Plot information
                    case = 9
                # Case 29
                else:  # speed_r == 7.5
                    # No speed change
                    # Correction filter: CCIRw7.5 + NABr7.5
                    # - CCIR_7.5 constant not altered
                    t2 = t2_7
                    # - NAB_7.5 constant not altered
                    t4 = t4_7
                    # Filter coefficients
                    a = [t3 * t4, t3, 0]
                    b = [t2 * t3, t2 + t3, 1]
                    # Plot information
                    case = 29
        else:  # speed_w == 3.75
            if standard_r == 'NAB':
                # Case 25
                if speed_r == 15:
                    fs = fs / 4  # Quartering the sampling frequency
                    # Correction filter: NABw15_mod + NABr3.75
                    # - NAB constants multiplied by 4
                    t3_mod = t3 * 4
                    t4_mod = t4_15 * 4
                    # - NABr constant not altered
                    t4 = t4_3
                    # Filter coefficients
                    a = [t3 * t3_mod * t4, t3 * (t3_mod + t4), t3]
                    b = [t3 * t3_mod * t4_mod, t3_mod * (t3 + t4_mod), t3_mod]
                    # Plot information
                    case = 25
                # Case 26
                elif speed_r == 7.5:
                    fs = fs / 2  # Halving the sampling frequency
                    # Correction filter: NABw7.5_mod + NABr3.75
                    # - NAB constants multiplied by 2
                    t3_mod = t3 * 2
                    t4_mod = t4_7 * 2
                    # - NABr constant not altered
                    t4 = t4_3
                    # Filter coefficients
                    a = [t3 * t3_mod * t4, t3 * (t3_mod + t4), t3]
                    b = [t3 * t3_mod * t4_mod, t3_mod * (t3 + t4_mod), t3_mod]
                    # Plot information
                    case = 26
                # Case 36
                else:  # speed_r == 3.75
                    print('Reference case: 36')
                    pprint('Nothing to do!', color=Color.GREEN)
                    return None
            else:  # standard_r == 'CCIR'
                # Case 12
                if speed_r == 30:
                    fs = fs / 8  # Dividing by 8 the sampling frequency
                    # Correction filter: CCIRw30_mod + NABr3.75
                    # - CCIR_30 constant multiplied by 8
                    t2 = t2_30 * 8
                    # - NAB_3.75 constant not altered
                    t4 = t4_3
                    # Filter coefficients
                    a = [t3 * t4, t3, 0]
                    b = [t2 * t3, t2 + t3, 1]
                    # Plot information
                    case = 12
                # Case 13
                elif speed_r == 15:
                    fs = fs / 4  # Quartering the sampling frequency
                    # Correction filter: CCIRw15_mod + NABr3.75
                    # - CCIR_15 constant multiplied by 4
                    t2 = t2_15 * 4
                    # - NAB_3.75 constant not altered
                    t4 = t4_3
                    # Filter coefficients
                    a = [t3 * t4, t3, 0]
                    b = [t2 * t3, t2 + t3, 1]
                    # Plot information
                    case = 13
                # Case 14
                else:  # speed_r == 7.5
                    fs = fs / 2  # Halving the sampling frequency
                    # Correction filter: CCIRw7.5_mod + NABr3.75
                    # - CCIR_7.5 constant multiplied by 2
                    t2 = t2_7 * 2
                    # - NAB_3.75 constant not altered
                    t4 = t4_3
                    # Filter coefficients
                    a = [t3 * t4, t3, 0]
                    b = [t2 * t3, t2 + t3, 1]
                    # Plot information
                    case = 14
    return a, b, fs, case


def correction(a: array, b: array, paf: ndarray, fs: int) -> ndarray:
    """
    Apply a correction filter to a Preservation Audio File;

    Parameters
    ----------
    a : array
        array of coefficients, specifying the numerator of filter transfer function,
    b : array
        array of coefficients, specifying in the denominator of filter transfer function,
    paf : ndarray
        specifying the raw audio data of the Preservation Audio File,
    fs : int
        specifying the operational sampling frequency.
        
    Returns
    -------
    ndarray
        the corrected audio as a Restored Audio File.
    """

    # Analog transfer function
    h_a = TransferFunction(a, b)

    # Digital transfer function through bilinear digitisation
    h_d = c2d(h_a, 1 / fs, 'bilinear')
    num_d = h_d.num[0][0]  # Inspect Hd.num to see why [0][0] is needed...
    den_d = h_d.den[0][0]  # Same story here

    # Pole check

    # New pole frequency
    pole_frequency = 2
    # Move to zero-pole representation
    z, p, k = tf2zpk(a, b)
    # Check if the function presents a pole at 0 Hz
    for i in range(len(p)):
        if p[i] == 0:
            # Replace pole
            p[i] = -pole_frequency * 2 * np.pi
            pprint('\nPole at 0 Hz replaced!', color=Color.PURPLE)
            # Back to transfer function representation
            ap, bp = zpk2tf(z, p, k)

            # Analog transfer function
            hp_a = TransferFunction(ap, bp)

            # Digital transfer function through bilinear digitisation
            hp_d = c2d(hp_a, 1 / fs, 'bilinear')
            num_d = hp_d.num[0][0]
            den_d = hp_d.den[0][0]

    print('\nFiltering Preservation Audio File...')
    # Filter Preservation Audio File
    raf = lfilter(num_d, den_d, paf, axis=0)
    # Again, wavfile.write() is stupid, and you must cast everything to not destroy your ears...
    raf = np.rint(raf).astype(paf.dtype)
    return raf


def save_file(file: ndarray, fs: int, temp_path: str, name: str):
    """
    Save an audio file to the given path with name 1.wav;

    Parameters
    ----------
    file : ndarray
        Array containing the raw audio data.
    fs : int
        Sampling frequency.
    temp_path : str
        Path where the file will be saved.
    name : str
        File name.
    
    Returns
    -------
    exit codes corresponding to the execution status.
    """
    raf_path = os.path.join(temp_path, 'RestoredAudioFiles')
    make_raf = False
    if not os.path.exists(raf_path):
        # Create directory
        os.mkdir(raf_path)
        make_raf = True
        print("Restored Audio Files directory '% s' created" % raf_path)
    else:
        pprint(f"Restored Audio Files directory '{raf_path}' already exists!", color=Color.PURPLE)
        overwrite = input('Do you want to overwrite it? [y/n]: ')
        if overwrite.casefold() == 'y':
            # Overwrite directory
            shutil.rmtree(raf_path)
            os.mkdir(raf_path)
            make_raf = True
            print('Restored Audio Files directory overwritten')
        elif overwrite.casefold() != 'n':
            pprint('Unknown command, exiting', color=Color.RED)
            quit(os.EX_USAGE)
    if make_raf:
        print("Saving Restored Audio File to: '%s' ..." % raf_path)
        wavfile.write(os.path.join(raf_path, name + '.wav'), fs, file)


def read_params_from_irregularity_file(irregularity_file_path: str) -> tuple[str, float, str, float]:
    """Get the parameters needed to apply the correction filter from the specified irregularity file.
    
    Parameters
    ----------
    irregularity_file_path : str
        Path to the irregularity file.
    
    Returns
    -------
    tuple[str, float, str, float]
        Tuple containing the parameters needed to apply the correction filter.
        It is assumed that the audio file considered contains only one recording session (so only one speed and equalisation).
        Since it is only one recording session, the possible correction to be applied is only one, this fact allow us to
        consider the mode of the equalisation and speed detected as the correct parameters.
        If there are no irregularities properties in the irregularity file, the parameters default to:
        - reading equalisation: NAB
        - reading speed: 7.5
        - writing equalisation: NAB
        - writing speed: 7.5  
    """
    json_content = File(irregularity_file_path, FileType.JSON).get_content()
    irregularity_file = IrregularityFile(**json_content)
    write_speeds = []
    read_speeds = []
    write_eq = []
    read_eq = []

    for irr in irregularity_file.irregularities:
        if irr.properties is not None:
            read_eq.append(irr.properties.reading_equalisation)
            read_speeds.append(irr.properties.reading_speed)
            write_eq.append(irr.properties.writing_equalisation)
            write_speeds.append(irr.properties.writing_speed)
        else:
            read_eq.append(None)
            read_speeds.append(None)
            write_eq.append(None)
            write_speeds.append(None)

    result_ws, result_rs, result_we, result_re = mode(write_speeds), mode(read_speeds), mode(write_eq), mode(read_eq)
    if all(x is None for x in [result_rs, result_ws]) or len(write_speeds) == 0:
        result_ws, result_rs = 7.5, 7.5
    if all(x is None for x in [result_we, result_re]) or len(write_eq) == 0:
        result_we, result_re = EqualizationStandard.NAB, EqualizationStandard.NAB

    return result_we, result_ws, result_re, result_rs
