import os
import sys
from argparse import ArgumentParser, RawTextHelpFormatter
from typing import NoReturn
from rich.console import Console
from scipy.io import wavfile

from mpai_cae_arp.audio.standards import SpeedStandard
from mpai_cae_arp.time import seconds_to_string
from mpai_cae_arp.types.restoration import Restoration, EditingList


from .lib import correction, get_correction_filter, save_file, check_input


def get_args() -> tuple[str, str, str, float, str, float]:
    """
    Method to obtain arguments from environment variables or command line.
    Default environment, ignored if a command line argument is passed.

    Raises
    ------
    ValueError
        If no arguments are passed and at least one environment variable is not set.

    Returns
    -------
    tuple[str, str, str, float, str, float]
        tuple consisting of six variables:

        * str specifying the working path;
        * str specifying the name of the Preservation files, which is key element to retrieve necessary files;
        * str specifying the equalization standard used when the tape was recorded;
        * float specifying the speed used when the tape was recorded;
        * str specifying the equalization standard used when the tape was read;
        * float specifying the speed used when the tape was read.
    """

    if len(sys.argv) > 1:
        # Read from command line
        parser = ArgumentParser(
            prog="python3 tapeAudioRestoration.py",
            formatter_class=RawTextHelpFormatter,
            description="A tool that implements MPAI CAE-ARP Tape Audio Restoration Technical Specification.\n"
                        "By default, the configuration parameters are loaded from the environment,\n"
                        "but, alternately, you can pass command line arguments to replace them."
        )
        parser.add_argument(
            "-w",
            "--working-path",
            help="Specify the Working Path, where all input files are stored",
            required=True
        )
        parser.add_argument(
            "-f",
            "--files-name",
            help="Specify the name of the Preservation files (without extension)",
            required=True
        )
        parser.add_argument(
            "-ew",
            "--equalization-w",
            help="Specify the name of the equalization standard used when the tape was recorded",
            required=True
        )
        parser.add_argument(
            "-sw",
            "--speed-w",
            help="Specify the speed used when the tape was recorded",
            required=True
        )
        parser.add_argument(
            "-er",
            "--equalization-r",
            help="Specify the name of the equalization standard used when the tape was read",
            required=True
        )
        parser.add_argument(
            "-sr",
            "--speed-r",
            help="Specify the speed used when the tape was read",
            required=True
        )
        args = parser.parse_args()
        working_path = args.working_path
        files_name = args.files_name
        standard_w = args.equalization_w
        speed_w = args.speed_w
        standard_r = args.equalization_r
        speed_r = args.speed_r
    else:
        arg_names = ['WORKING_PATH', 'FILES_NAME', 'STANDARD_W', 'SPEED_W', 'STANDARD_R', 'SPEED_R']
        config = {argument: os.getenv(argument) for argument in arg_names}
        working_path, files_name, standard_w, speed_w, standard_r, speed_r = map(config.get, arg_names)
        
        if any(value is None for value in config.values()):
            raise ValueError("Please, set the environment variables: WORKING_PATH, FILES_NAME, STANDARD_W, SPEED_W, STANDARD_R, SPEED_R.")

    return working_path, files_name, standard_w, float(speed_w), standard_r, float(speed_r)


def exit_with_error(error_message: str, error: int, console: Console) -> NoReturn:
    console.print(f"[red bold]Error: {error_message}")
    quit(error)


def main():
    console = Console()

    console.print("[bold]\nWelcome to ARP Tape Audio Restoration!")
    console.print(f"You are using Python version: {sys.version}")

    try:
        working_path, files_name, standard_w, speed_w, standard_r, speed_r = get_args()
        paf_path, temp_path, standard_w, standard_r = check_input(working_path, files_name, standard_w, speed_w, standard_r, speed_r)
        
        # open the Preservation Audio file
        with console.status("[bold]Opening Preservation Audio file...", spinner="dots"):
            sample_rate, preservation_audio_file = wavfile.read(paf_path)
        # get the correction filter
        with console.status("[bold]Applying correction filter...", spinner="dots"):
            correction_filter = get_correction_filter(standard_w, speed_w, standard_r, speed_r, sample_rate)

            restored_audio_file = preservation_audio_file
            if correction_filter is None:
                new_sample_rate = sample_rate
            else:
                a, b, new_sample_rate, _ = correction_filter
                # if needed, apply the correction filter to the Preservation Audio file
                if len(a) != 0:
                    restored_audio_file = correction(a, b, preservation_audio_file, round(new_sample_rate))
        with console.status("[bold]Saving Restoration Audio file...", spinner="dots"):
            save_file(restored_audio_file, round(new_sample_rate), temp_path, files_name)

        
        editing_list_path = os.path.join(working_path, 'temp', files_name, 'EditingList.json' )
        editing_list = EditingList(
            OriginalEqualisationStandard=standard_w,
            OriginalSampleFrequency=sample_rate,
            OriginalSpeedStandard=SpeedStandard(speed_w),
            Restorations=[
                Restoration(
                    PreservationAudioFileStart="00:00:00.000",
                    PreservationAudioFileEnd=seconds_to_string(len(preservation_audio_file) / sample_rate),
                    RestoredAudioFileURI=f"{temp_path}/RestoredAudioFile/{files_name}.wav",
                    ReadingBackwards=False,
                    AppliedSpeedStandard=SpeedStandard(speed_r),
                    AppliedSampleFrequency=new_sample_rate,
                    AppliedEqualisationStandard=standard_r,
                )
            ]
        )

        editing_list.save_as_json_file(editing_list_path)

    except ValueError:
        exit_with_error(
            "{}\n{}".format(
                "Working directory or files name not specified!",
                "Try -h/--help to know more about Tape Audio Restoration usage"),
            os.EX_USAGE, console)
    except Exception as e:
        exit_with_error(str(e), os.EX_SOFTWARE, console)

    console.print("[green] Success! :tada:")
    quit(os.EX_OK)
