import subprocess
import json
import yaml
from fastapi import FastAPI, Response, status

from tapeAudioRestoration import __author__, __version__, __license__, __email__, __maintainer__

app = FastAPI(
    title="Tape Audio Restoration",
    description="""
[![MPAI CAE-ARP](https://img.shields.io/badge/MPAI%20CAE--ARP-gray?style=for-the-badge&logo=AppleMusic&logoColor=cyan&link=https://mpai.community/standards/mpai-cae/about-mpai-cae/)](https://mpai.community/standards/mpai-cae/about-mpai-cae/)

Implements the Technical Specification of [MPAI CAE-ARP](https://mpai.community/standards/mpai-cae/about-mpai-cae/#Figure2) *Tape Audio Restoration* AIM, providing:
* Restored Audio Files;
* Editing List.
""",
    version=__version__,
    contact={
        "name": __maintainer__,
        "email": __email__
    },
    license_info={
        "name": __license__,
    }
)


@app.get("/")
def index():
    return {"endpoints": ["restore", "docs"]}


@app.get("/restore", status_code=200)
async def restore(files_name: str, equalization_w: str, equalization_r: str, speed_w: str, speed_r: str, response: Response, working_path: str | None = None):
    
    process = [
        "python",
        "tapeAudioRestoration.py",
        "--files_name", files_name,
        "--equalization-w", equalization_w,
        "--equalization-r", equalization_r,
        "--speed-w", speed_w,
        "--speed-r", speed_r
        ]

    if working_path is not None:
        process.extend(["--working_path", working_path])
    else:
        with open("config.yaml") as yaml_file:
            config = yaml.safe_load(yaml_file)
        process.extend(["--working_path", config["WorkingPath"]])
        

    cprocess = subprocess.run(process, capture_output=True)

    if cprocess.returncode == 0:
        
        with open('EditingList.json') as json_file:
            editing_list = json.load(json_file)
        return editing_list

    else:
        response.status_code = status.HTTP_412_PRECONDITION_FAILED
        return {"error": {
            "returncode": cprocess.returncode,
            "stdout": cprocess.stdout.decode("utf-8"),
            "stderr": cprocess.stderr.decode("utf-8"),
        }}
