import subprocess
import json
from fastapi import FastAPI, Response, status

from mpai_cae_arp.types.schema import Info
from mpai_cae_arp.files import get_file_content

from src.tapeAudioRestoration import __author__, __version__, __license__, __email__, __maintainer__

info = Info(
    title="Tape Audio Restoration",
    description="""
[![MPAI CAE-ARP](https://img.shields.io/badge/MPAI%20CAE--ARP-gray?style=for-the-badge&logo=AppleMusic&logoColor=cyan&link=https://mpai.community/standards/mpai-cae/about-mpai-cae/)](https://mpai.community/standards/mpai-cae/about-mpai-cae/)

Implements the Technical Specification of [MPAI CAE-ARP](https://mpai.community/standards/mpai-cae/about-mpai-cae/#Figure2) *Tape Audio Restoration* AIM, providing:
* Restored Audio Files;
* Editing List.
""",
    version=__version__,
    contact={
        "name": __maintainer__,
        "email": __email__
    },
    license_info={"name": __license__, "url": "https://www.gnu.org/licenses/gpl-3.0.html"}
)


app = FastAPI(**(json.loads(info.json())))


@app.get("/")
def list_all_entrypoints() -> list[str]:
    return ["restore", "description", "docs"]


@app.get("/description")
async def get_server_info() -> Info:
    return info


@app.get("/restore", status_code=200)
async def restore(files_name: str, equalization_w: str, equalization_r: str, speed_w: str, speed_r: str, response: Response, working_path: str | None = None):
    
    process = [
        "python",
        "src/tapeAudioRestoration.py",
        "--files_name", files_name,
        "--equalization-w", equalization_w,
        "--equalization-r", equalization_r,
        "--speed-w", speed_w,
        "--speed-r", speed_r
        ]

    if working_path is not None:
        process.extend(["--working_path", working_path])
    else:
        config = get_file_content("config/args.yaml", "yaml")
        process.extend(["--working_path", config["WORKING_PATH"]])
        

    cprocess = subprocess.run(process, capture_output=True)

    if cprocess.returncode == 0:
        
        editing_list = get_file_content("EditingList.json", "json")
        return editing_list

    else:
        response.status_code = status.HTTP_412_PRECONDITION_FAILED
        return {"error": {
            "returncode": cprocess.returncode,
            "stdout": cprocess.stdout.decode("utf-8"),
            "stderr": cprocess.stderr.decode("utf-8"),
        }}
