import os
import shutil
from concurrent import futures
import grpc
import json
from grpc import StatusCode
from rich.console import Console
from moviepy.editor import VideoFileClip, AudioFileClip

from mpai_cae_arp.files import File, FileType
from mpai_cae_arp.network import arp_pb2_grpc
from mpai_cae_arp.network.arp_pb2 import (
    JobRequest,
    JobResponse,
    Contact,
    InfoResponse,
    License,
)
from packager import lib

PORT = os.getenv("PORT") or '50051'
info = File('config.yml', FileType.YAML).get_content()


def error_response(context, status, message):
    context.set_code(status)
    context.set_details(message)
    return JobResponse(status="error", message=message)


class PackagerServicer(arp_pb2_grpc.AIMServicer):

    def __init__(self, console: Console):
        self.console = console

    def getInfo(self, request, context) -> InfoResponse:
        self.console.log('Received request for AIM info')

        context.set_code(StatusCode.OK)
        context.set_details('Success')

        return InfoResponse(
            title=info['title'],
            description=info['description'],
            version=info['version'],
            contact=Contact(
                name=info['contact']['name'],
                email=info['contact']['email'],
            ),
            license=License(
                name=info['license_info']['name'],
                url=info['license_info']['url'],
            )
        )

    def work(self, request: JobRequest, context):
        self.console.log('Received request for computation')
        self.console.log(request)

        working_dir: str = request.working_dir
        files_name: str = request.files_name

        temp_dir = os.path.join(working_dir, "temp", files_name)

        yield JobResponse(status="success", message="Job started")

        # By default, AccessCopyFiles directory is created in the output path...
        acf_dir = 'AccessCopyFiles/' + files_name + '/'
        acf_path = os.path.join(working_dir, acf_dir)
        # ...however, if it already exists, the user can decide if it has to be overridden
        make_acf = lib.make_dir(acf_path)

        if make_acf:
            # Copy RestoredAudioFiles
            raf_path = os.path.join(temp_dir, 'RestoredAudioFiles')
            if not os.path.exists(raf_path):
                yield error_response(context, StatusCode.INVALID_ARGUMENT, "Restored Audio Files directory not found")
            restored_audio_files = os.listdir(raf_path)
            if len(restored_audio_files) == 1:
                yield JobResponse(status="warning", message="Restored Audio Files directory is empty")
            shutil.copytree(raf_path, os.path.join(acf_path, 'RestoredAudioFiles'))
            console.log("Restored Audio Files copied")
            yield JobResponse(status="success", message="Restored Audio Files copied")

            # Copy Editing List
            el_path = os.path.join(temp_dir, 'EditingList.json')
            try:
                shutil.copy2(el_path, acf_path)
            except FileNotFoundError:
                yield error_response(context, StatusCode.INVALID_ARGUMENT, "Editing List file not found")
            console.log("Editing List copied")
            yield JobResponse(status="success", message="Editing List copied")

            # Create Irregularity Images archive
            ii_path = os.path.join(temp_dir, 'IrregularityImages')
            if not os.path.exists(ii_path):
                yield error_response(context, StatusCode.INVALID_ARGUMENT, "Irregularity Images directory not found")
            irregularity_images = os.listdir(ii_path)
            if len(irregularity_images) == 1:
                yield JobResponse(status="warning", message="Irregularity Images directory is empty")
            shutil.make_archive(acf_path + 'IrregularityImages', 'zip', temp_dir, 'IrregularityImages')
            console.log("Irregularity Images archive created")
            yield JobResponse(status="success", message="Irregularity Images archive created")

            # Copy Irregularity File
            if_path = os.path.join(temp_dir, 'TapeIrregularityClassifier_IrregularityFileOutput2.json')
            try:
                shutil.copy2(if_path, os.path.join(acf_path, 'IrregularityFile.json'))
            except FileNotFoundError:
                yield error_response(context, StatusCode.INVALID_ARGUMENT, "Irregularity File file not found")
            console.log("Irregularity File copied")
            yield JobResponse(status="success", message="Irregularity File copied")

        # Preservation Master Files
        console.log('Creation of Preservation Master Files...')
        yield JobResponse(status="success", message="Creation of Preservation Master Files...")

        # By default, PreservationMasterFiles directory is created in the output path...
        pmf_dir = 'PreservationMasterFiles/' + files_name + '/'
        pmf_path = os.path.join(working_dir, pmf_dir)
        # ...however, if it already exists, it is possible to skip its creation
        make_pmf = lib.make_dir(pmf_path)

        if make_pmf:

            # Copy Preservation Audio File
            audio_file = files_name + '.wav'
            paf_path = os.path.join(working_dir, 'PreservationAudioFile/', audio_file)
            try:
                shutil.copy2(paf_path, pmf_path + 'PreservationAudioFile.wav')
            except FileNotFoundError:
                yield error_response(context, StatusCode.INVALID_ARGUMENT, "Preservation Audio File file not found")
            console.log("Preservation Audio File copied")
            yield JobResponse(status="success", message="Preservation Audio File copied")

            # Create Preservation Audio-Visual File with substituted audio
            video_file = files_name + '.mov'
            pvf_path = os.path.join(working_dir, 'PreservationAudioVisualFile/', video_file)
            try:
                audio = AudioFileClip(paf_path)
                video = VideoFileClip(pvf_path)
                # Open Irregularity File to get offset
                irregularity_file_json = open(
                    os.path.join(temp_dir, 'TapeIrregularityClassifier_IrregularityFileOutput2.json')
                )
                irregularity_file = json.load(irregularity_file_json)
                offset = irregularity_file['Offset']/1000
                if offset > 0:
                    audio = audio.subclip(t_start=offset)
                else:
                    video = video.subclip(t_start=offset)
                video = video.set_audio(audio)
                video.write_videofile(pmf_path + 'PreservationAudioVisualFile.mov', bitrate='3000k', codec='mpeg4')
                console.log("Preservation Audio-Visual File created")
                yield JobResponse(status="success", message="Preservation Audio-Visual File created")
            except OSError:
                yield error_response(context, StatusCode.INVALID_ARGUMENT, "Preservation Audio-Visual File file not found")

            # Create Irregularity Images archive (already checked)
            shutil.make_archive(pmf_path + 'IrregularityImages', 'zip', temp_dir, 'IrregularityImages')
            console.log("Irregularity Images archive created")
            yield JobResponse(status="success", message="Irregularity Images archive created")

            # Copy Irregularity File (already checked)
            shutil.copy2(
                os.path.join(temp_dir, 'TapeIrregularityClassifier_IrregularityFileOutput2.json'),
                os.path.join(pmf_path, 'IrregularityFile.json')
            )
            console.log("Irregularity File copied")
            yield JobResponse(status="success", message="Irregularity File copied")

            # End Preservation Master Files
            yield JobResponse(status="success", message="Success!")

        else:
            console.log("\nExit")
            yield JobResponse(status="success", message="Exit")


def serve(console):
    server = grpc.server(futures.ThreadPoolExecutor(max_workers=10))
    arp_pb2_grpc.add_AIMServicer_to_server(PackagerServicer(console), server)
    server.add_insecure_port(f'[::]:{PORT}')
    server.start()
    server.wait_for_termination()


if __name__ == '__main__':
    console = Console()
    console.print(f'Server started at localhost:{PORT} :satellite:')
    serve(console)
