import argparse
import os
import time
from rich.console import Console
import segment_finder as sf

from mpai_cae_arp.types.irregularity import IrregularityFile


def get_args() -> argparse.Namespace:
    parser = argparse.ArgumentParser()
    parser.add_argument("--working-directory", "-w", help="Working directory", required=True)
    parser.add_argument("--files-name", "-f", nargs="+", help="Files name", required=True)
    return parser.parse_args()


def exit_with_error(error_message: str, console) -> None:
    console.print(f"[red bold]Error: {error_message}")
    quit(os.EX_USAGE)


def main():
    args = get_args()
    console = Console()
    console.print("[bold]Welcome to ARP Audio Analyzer!")
    
    working_directory = args.working_directory
    files_name = args.files_name

    with console.status("Reading input files...", spinner="dots"):
        time.sleep(1)
        audio_src = os.path.join(working_directory, "PreservationAudioFile", f"{files_name[0]}.wav")
        video_src = os.path.join(working_directory, "PreservationAudioVisualFile", f"{files_name[0]}.mov")

        console.log(audio_src)
        console.log(video_src)
        
        audio_exists = os.path.exists(audio_src)
        video_exists = os.path.exists(video_src)

        match audio_exists, video_exists:
            case True, True:
                console.print("[green]Input files found!")
            case False, True:
                exit_with_error("Audio file not found!", console)
            case True, False:
                exit_with_error("Video file not found!", console)
            case False, False:
                exit_with_error("Input files not found!", console)

    # create irregularity file 1
    with console.status("Creating irregularity file 1...", spinner="dots"):
        irreg1 = sf.create_irreg_file(audio_src, video_src)

    # create irregularity file 2
    with console.status("Creating irregularity file 2...", spinner="dots"):
        video_irreg_1 = {}
        irreg2 = sf.merge_irreg_files(irreg1, IrregularityFile.from_json(video_irreg_1))

    with console.status("Extracting audio irregularities...", spinner="bouncingBall"):
        sf.extract_audio_irregularities(audio_src, irreg2, working_directory)

    # classify audio irregularities

    console.print("[green bold]Success!")


if __name__ == "__main__":
    main()
