import tempfile
import pytest
import json
from mpai_cae_arp.audio.standards import EqualizationStandard, SpeedStandard
from mpai_cae_arp.types.restoration import Restoration, EditingList


class TestRestoration:

    correct_data = {
        "RestorationID": "00000000-0000-0000-0000-000000000000",
        "PreservationAudioFileStart": "00:00:00.000",
        "PreservationAudioFileEnd": "00:00:10.000",
        "RestoredAudioFileURI": "https://www.google.com",
        "ReadingBackwards": False,
        "AppliedSpeedStandard": 15,
        "AppliedSampleFrequency": 44100,
        "AppliedEqualisationStandard": "IEC1"
    }

    my_rest = Restoration(RestorationID="00000000-0000-0000-0000-000000000000",
                          PreservationAudioFileEnd="00:00:10.000",
                          PreservationAudioFileStart="00:00:00.000",
                          ReadingBackwards=False,
                          AppliedSampleFrequency=44100,
                          AppliedSpeedStandard=SpeedStandard.V,
                          AppliedEqualisationStandard=EqualizationStandard.CCIR,
                          RestoredAudioFileURI="https://www.google.com")

    def test_init_from_dict(self):
        assert Restoration(**self.correct_data) == self.my_rest

    def test_init_from_object(self):
        assert Restoration(**self.my_rest.dict()) == self.my_rest

    def test_init_from_json(self):
        assert Restoration.parse_raw(self.my_rest.json()) == self.my_rest

    def test_json_serialize(self):
        assert '"id": "00000000-0000-0000-0000-000000000000"' in self.my_rest.json()
        assert '"RestorationID": "00000000-0000-0000-0000-000000000000"' in self.my_rest.json(
            by_alias=True)


class TestEditingList:

    data = {
        "OriginalSpeedStandard":
        15,
        "OriginalEqualisationStandard":
        "IEC1",
        "OriginalSampleFrequency":
        44100,
        "Restorations": [{
            "RestorationID": "00000000-0000-0000-0000-000000000000",
            "PreservationAudioFileStart": "00:00:00.000",
            "PreservationAudioFileEnd": "00:00:10.000",
            "RestoredAudioFileURI": "https://www.google.com",
            "ReadingBackwards": False,
            "AppliedSpeedStandard": 15,
            "AppliedSampleFrequency": 44100,
            "AppliedEqualisationStandard": "IEC1"
        }]
    }

    rest = Restoration(PreservationAudioFileStart="00:00:00.000",
                       PreservationAudioFileEnd="00:00:10.000",
                       RestoredAudioFileURI="https://www.google.com",
                       ReadingBackwards=False,
                       AppliedSampleFrequency=44100,
                       AppliedSpeedStandard=SpeedStandard.V,
                       AppliedEqualisationStandard=EqualizationStandard.CCIR)

    my_editing_list: EditingList = EditingList(
        OriginalEqualisationStandard=EqualizationStandard.CCIR,
        OriginalSampleFrequency=44100,
        OriginalSpeedStandard=SpeedStandard.V,
        Restorations=[])\
            .add(Restoration(
                    RestorationID="00000000-0000-0000-0000-000000000000",
                    PreservationAudioFileStart="00:00:00.000",
                    PreservationAudioFileEnd="00:00:10.000",
                    RestoredAudioFileURI="https://www.google.com",
                    ReadingBackwards=False,
                    AppliedSpeedStandard=SpeedStandard.V,
                    AppliedSampleFrequency=44100,
                    AppliedEqualisationStandard=EqualizationStandard.CCIR
                )
            )

    def test_init_from_dict(self):
        assert EditingList(**self.data) == self.my_editing_list

    def test_init_from_object(self):
        assert EditingList(**self.my_editing_list.dict()) == self.my_editing_list

    def test_init_from_json(self):
        assert EditingList.parse_raw(
            self.my_editing_list.json()) == self.my_editing_list

    def test_add(self):
        tmp: EditingList = self.my_editing_list.copy(deep=True)
        tmp.add(self.rest)

        assert len(tmp.restorations) == 2

    def test_remove(self):
        tmp: EditingList = self.my_editing_list\
                            .copy(deep=True)\
                            .remove(self.my_editing_list.restorations[0])

        assert len(tmp.restorations) == 0

        with pytest.raises(ValueError):
            tmp.remove(self.rest)

    def test_remove_by_id(self):
        tmp: EditingList = self.my_editing_list\
                            .copy(deep=True)\
                            .remove_by_id(self.my_editing_list.restorations[0].id)

        assert len(tmp.restorations) == 0

        with pytest.raises(ValueError):
            tmp.remove_by_id(self.rest.id)

    def test_save_as_json_file(self):
        tmp = tempfile.NamedTemporaryFile(delete=False)
        tmp.close()
        
        self.my_editing_list.save_as_json_file(tmp.name)
        with open(tmp.name, 'r') as f:
            tmp_content = json.load(f)
            assert self.my_editing_list.json(by_alias=True, indent=4) == tmp_content
