import uuid
from typing import TypeVar
from pydantic import BaseModel, Field
from mpai_cae_arp.audio.standards import SpeedStandard, EqualizationStandard
from mpai_cae_arp.files import File, FileType


class Restoration(BaseModel):
    id: uuid.UUID = Field(default_factory=uuid.uuid4)
    preservation_audio_file_start: str
    preservation_audio_file_end: str
    restored_audio_file_URI: str
    reading_backwards: bool
    applied_speed_standard: SpeedStandard
    applied_sample_frequency: int
    original_equalization_standard: EqualizationStandard


Self = TypeVar("Self", bound="EditingList")


class EditingList(BaseModel):
    """
    .. versionadded:: 0.4.0
    """
    original_speed_standard: SpeedStandard
    original_equalization_standard: EqualizationStandard
    original_sample_frequency: int
    restorations: list[Restoration]

    def add(self, restoration: Restoration) -> Self:
        self.restorations.append(restoration)
        return self

    def remove(self, restoration: Restoration) -> Self:
        self.restorations.remove(restoration)
        return self

    def remove_by_id(self, restoration_id: uuid.UUID) -> Self:
        filtered = list(filter(lambda r: r.id != restoration_id, self.restorations))
        self.restorations = filtered
        return self

    def save_as_json_file(self, path: str) -> None:
        File(path=path, file_type=FileType.JSON).write_content(self.json())


if __name__ == "__main__":
    rest = Restoration(preservation_audio_file_start="00:00:00.000",
                       preservation_audio_file_end="00:00:10.000",
                       restored_audio_file_URI="https://www.google.com",
                       reading_backwards=False,
                       applied_sample_frequency=44100,
                       applied_speed_standard=SpeedStandard.III,
                       original_equalization_standard=EqualizationStandard.CCIR)

    editing_list = EditingList(original_equalization_standard=EqualizationStandard.CCIR,
                               original_speed_standard=SpeedStandard.III,
                               original_sample_frequency=44100,
                               restorations=[])

    editing_list.add(rest)

    print(rest)
    print(editing_list.json())
